# DACS-2500KB PWMパルス出力

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# Drsw_rpi.py と FT_rpi.py を同じディレクトリに格納

import FT_rpi
import math

finterclk = 1.0               # クロック周波数設定(MHz)
internum = 20000              # 周期(クロック数)
interrdy = 0                  # クロック周波数/周期設定済

def Drsw_init(devnum):
# 関数 デバイスのOPEN
#    引数   : デバイス番号(int型)
#    戻り値 : (1)'OK' 正常終了 'NG' 異常 ,(2)デバイスハンドル

    ret = 'NG'            # 仮に異常終了とする
    for cnt in range(4):  # 4デバイス分の検索
        if devnum >= 9:
            devn = cnt
        else:             # デバイス番号を指定している
            devn = devnum
        handle = FT_rpi.init_dacs(devn)
        if handle.value != None:
            # デジタル入力を実行
            readdata = FT_rpi.transfer_dacs(handle, 'W0Rxxxxx'+chr(0xd), 9)
            if (len(readdata) == 9) and (readdata[0:1] == 'R'):
                # qコマンドの応答を確認(PWMパルス出力)
                readdata = FT_rpi.transfer_dacs(handle, 'q0R'+chr(0xd), 9)
                if (len(readdata) == 9) and (readdata[0:1] == 'n'):
                    ret = 'OK'
                readdata = FT_rpi.transfer_dacs(handle, 'W0Rxxxxx'+chr(0xd), 9)
                if ret == 'OK':
                    break
            FT_rpi.close_dacs(handle)   # デバイスをCLOSE
        if devnum < 9:
            break
    if ret == 'NG':
        devn = 9
    return ret,handle

def Drsw_read(handle, chno):
# 関数 パルス幅の取得
#    引数   : デバイスハンドル, 
#             チャンネル番号  文字列 x(0～11)  12は全ch指定
#    戻り値 : 文字列 パルス幅値
#                    全ch指定のときはch0からch11をカンマ区切り 
    sdata = ''
    try:
        ci = int(chno)            # チャンネル番号
    except ValueError:
        ci = 99

    # 単一チャンネル指定のとき
    if ci >= 0 and ci < 12:
        dcommand = 'Q00' + format(ci, 'x') + 'R' + chr(0xd)
        # コマンド送信とパルス幅受信
        readdata = FT_rpi.transfer_dacs(handle, dcommand, 9)
        if len(readdata) == 9 and readdata[0:1] == 'N':
            sdata = readdata[4:8]     # パルス幅

    # 全チャンネル指定のとき
    elif ci == 12:
        dcommand = 'Q000R&Q001R&Q002R&Q003R&Q004R&Q005R&'
        dcommand = dcommand + 'Q006R&Q007R&Q008R&Q009R&Q00AR&Q00BR'+ chr(0xd)
        # コマンド送信とパルス幅受信
        readdata = FT_rpi.transfer_dacs(handle, dcommand, 108)
        if len(readdata) == 108 and readdata[0:1] == 'N':
            for cnt in range(12):
                p = cnt*9 + 4
                sdata = sdata + readdata[(p):(p+4)] + ','   # パルス幅
            sdata = sdata[:-1]
    return sdata

def Drsw_sread(handle):
# 関数 ステータスとデジタル入力の取得
#    引数   : デバイスハンドル, 
#    戻り値 : 文字列 パルス幅変化状態(16進数3桁), デジタル入力(16進数6桁)
    sdata = ''
    dcommand = 'q0R&W0R' + chr(0xd)
    # パルス幅変化状態とデジタル入力受信
    readdata = FT_rpi.transfer_dacs(handle, dcommand, 18)
    if len(readdata) == 18 and readdata[0:1] == 'n' and readdata[9:10] == 'R':
        sdata = readdata[5:8] + ',' + readdata[11:17]
    return sdata

def Drsw_iread():
# 関数 クロック周波数と周期の取得
#    引数   : なし
#    戻り値 : 実数値 クロック周波数,  整数値 周期(クロック数)

    global finterclk                  # クロック周波数(MHz)
    global internum                   # 周期(クロック数)
    global interrdy                   # クロック周波数/周期設定状態
    if interrdy == 1:
        fret = finterclk              # クロック周波数/周期設定済のとき
    else:
        fret = 99.0                   # クロック周波数/周期未設定のとき
    return fret,internum

def Drsw_write(handle, wdata):
# 関数 コマンド送信とレスポンス受信
#    引数   : デバイスハンドル, 指示データ
#    戻り値 : 'OK' 正常終了 'NG' 異常終了
#       データ連結では、一部のデータが不正のとき、すべての転送なし

# 指示データ(文字列)
# /(スラッシュ)で複数の指示データを連結可能  例 W,0,100/W,1,200/G

#     G       パルス出力開始
#     N       パルス出力終了
#     I,y--y,z--z  周期設定
#             y--y  クロック周波数設定(MHz) 下記のいずれか
#                    0.5, 1, 2, 4, 8, 16, 32, 64
#             z--z  周期設定 小数点可 (単位: μs)
#     IP,y--y,z--z  周期設定
#             y--y  クロック周波数設定(MHz) 下記のいずれか
#                    0.5, 1, 2, 4, 8, 16, 32, 64
#             z--z  周期設定 (単位: クロック数)
#     W,xx,y--y  目標パルス幅設定
#             xx   チャンネル番号 (0～11)  全チャンネル指定(12)
#             y--y パルス幅 小数点可 (単位: μs)
#     WP,xx,y--y  目標パルス幅設定
#             xx   チャンネル番号 (0～11)  全チャンネル指定(12)
#             y--y パルス幅 (単位: クロック数)
#     V,xx,y--y  変化速度設定
#             xx   チャンネル番号 (0～11)  全チャンネル指定(12)
#             y--y 変化速度 小数点可 (単位: パルス幅 μs / s)
#     VP,xx,y--y  変化速度設定
#             xx   チャンネル番号 (0～11)  全チャンネル指定(12)
#             y--y 変化速度 (単位: クロック数 / s)
#     S       パルス幅変化停止
#               停止後のパルス幅を目標パルス幅とし、停止を解除
#     DP,y--y デジタル出力極性(16進数6桁 bit23～0)
#     DO,y--y デジタル出力(16進数6桁 bit23～0)

    global finterclk                # クロック周波数(MHz)
    global internum                 # 周期(クロック数)
    global interrdy                 # クロック周波数/周期設定状態

    cdatalist = wdata.split('/')    # '/' で分離
    ret = 'NG'                      # 仮に異常終了とする
    alltdata = ''                   # コマンド文字列の準備

    for cntlist in range(len(cdatalist)):  # 指示データ数分の繰返し
        cdata = cdatalist[cntlist].split(',')     # ',' で分離
        clen = len(cdata)           # ','区切りのデータ数
        if len(cdata[0]) != 0:
            ca = (cdata[0]).upper()
            c1 = ca[0:1]            # 最初の1文字
        else:
            alltdata = ''
            break

        # コマンド文字列に変換

        tdata = ''                  # 1指示データ分のコマンド文字列の準備

        if c1 == 'G':               # パルス出力開始
            tdata = 'Q00F0000&'
        elif c1 == 'N':             # パルス出力停止
            tdata = 'Q00E0000&'
        elif c1 == 'S':             # パルス出力変化停止
            tdata = 'q00S' + chr(0xd)             # 変化停止
            readdata = FT_rpi.transfer_dacs(handle, tdata, 9)
            tdata = ''
            if len(readdata) == 9:
                readdata = Drsw_read(handle, '12')
                if len(readdata) == 59:
                    pnum = readdata.split(',')    # ',' で分離
                    for cnt in range(12):         # 現在のパルス幅を目標パルス幅にする
                        tdata = tdata + 'Q00' + format(cnt,'X') + pnum[cnt] + '&'
                    tdata = tdata + 'Q00F0000&'   # 停止解除
        elif clen == 2 and c1 == 'D':
            if ca == 'DP':                    # デジタル出力極性設定
                if len(cdata[1]) == 6:
                    tdata = 'y0'+ cdata[1] + '&'
            elif ca == 'DO':                  # デジタル出力(bit23～0)設定
                if len(cdata[1]) == 6:
                    tdata = 'W0' + cdata[1] + '&'
            else:
                alltdata = ''
                break

        elif clen == 3 and c1 == 'I':         # 周期設定
            pf = float(cdata[1]) + 0.01
            pc = 0.5
            cnum = 8
            for cnt in range(7):
                if pf < pc*2:
                    break
                pc = pc*2            # クロック周波数更新
                cnum = cnum + 1      # クロック周波数コード更新
            if ca == 'IP':
                try:
                    inum = int(cdata[2])-1    # 周期(クロック数)
                except ValueError:
                    alltdata = ''
                    break
            else:
                try:
                    ifdata = float(cdata[2])  # 周期(単位: μs)
                    inum = int(ifdata*pc)-1   # 周期(クロック数)
                except ValueError:
                    alltdata = ''
                    break
            if inum < 1:
                inum = 1
            if inum > 1048575:
                inum = 1048575
            finterclk = pc           # クロック周波数
            internum  = inum + 1     # 周期(クロック数)
            tdata = 'Q0' + format(cnum,'X') + format(inum,'05X') + '&'
            interrdy = 1             # クロック周波数/周期設定済とする

        elif clen == 3 and c1 == 'W':         # パルス幅設定
            try:
                chdata = int(cdata[1])        # チャンネル番号
            except ValueError:
                alltdata = ''
                break
            if ca == 'WP':
                try:
                    inum = int(cdata[2])      # パルス幅(クロック数)
                except ValueError:
                    alltdata = ''
                    break
            else:
                if interrdy == 0:             # クロック周波数/周期未設定のとき
                    alltdata = ''
                    break
                try:
                    pfdata = float(cdata[2])  # パルス幅(単位: μs)
                    inum = int(pfdata * finterclk)  # パルス幅(クロック数)
                except ValueError:
                    alltdata = ''
                    break
            if inum < 0:
                inum = 0
            if inum > 65535:
                inum = 65535
            if chdata >= 0 and chdata < 12:
                tdata = 'Q00' + format(chdata,'X') + format(inum,'04X') + '&'
            elif chdata == 12:
                for cnt in range(12):
                    tdata = tdata + 'Q00' + format(cnt,'X') + format(inum,'04X') + '&'

        elif clen == 3 and c1 == 'V':         # 変化速度設定
            try:
                chdata = int(cdata[1])        # チャンネル番号
            except ValueError:
                alltdata = ''
                break
            if ca == 'VP':
                try:                           # 変化速度(クロック数 / s)
                    vfdata = float(cdata[2])/finterclk  # 変化速度(単位: μs / s)
                except ValueError:
                    alltdata = ''
                    break
            else:
                if interrdy == 0:             # クロック周波数/周期未設定のとき
                    alltdata = ''
                    break
                try:
                    vfdata = float(cdata[2])   # 変化速度(単位: μs / s)
                except ValueError:
                    alltdata = ''
                    break
            if vfdata < 0:
                vfdata = -vfdata
            if vfdata < 1.0:
                vnum = 65535
            else:
                vnum = int(100000 / vfdata)
            if vnum < 0:
                vnum = 0
            if vnum > 65535:
                vnum = 65535
            if chdata >= 0 and chdata < 12:
                tdata = 'q00' + format(chdata,'X') + format(vnum,'04X') + '&'
            elif chdata == 12:
                for cnt in range(12):
                    tdata = tdata + 'q00' + format(cnt,'X') + format(vnum,'04X') + '&'
        if len(tdata) != 0:
            alltdata = alltdata + tdata     # コマンド文字列の連結
        else:
            alltdata = ''
            break

    # 正常な入力データのときコマンドを送信
    if len(alltdata) != 0:
        num = alltdata.count('&')       # コマンド数
        alltdata = alltdata[:-1] + chr(0xd)  # 末尾の'&'文字をCRに置換
        readdata = FT_rpi.transfer_dacs(handle, alltdata, num*9)
        if len(readdata) == num*9:
            ret = 'OK'                  # 正常終了とする
    return ret

def Drsw_pmove(handle, pdata):
# 関数 複数チャンネルの同時スタート&ストップ
#    引数   : デバイスハンドル, 指示データ
#    戻り値 : 'OK' 正常終了 'NG' 異常終了

# 指示データ(文字列)  'L/z--z/xx,y--y/xx,y--y/  --  /x,y--y'
#                        z--z  変化速度 (単位: μs / s) 小数点可
#                        xx    チャンネル番号 0～11
#                        y--y  目標パルス幅 (単位: μs) 小数点可
#                        対象となるチャンネルのデータを /（スラッシュ）で連結

# 指示データ(文字列)  'LP/z--z/xx,y--y/xx,y--y/  --  /x,y--y'
#                        z--z  変化速度 (単位: クロック数 / s)
#                        xx    チャンネル番号 0～11
#                        y--y  目標パルス幅 (単位: クロック数)
#                        対象となるチャンネルのデータを /（スラッシュ）で連結

    ret = 'NG'                      # 仮に異常終了とする
    if pdata[0:1].upper() != 'L':
        return ret                  # Lコマンドではない

    cdatalist = pdata.split('/')    # '/' で分離
    cdatanum = len(cdatalist) - 2   # 対象チャンネル数
    if cdatanum < 1:
        return ret                  # データ不足
    if cdatalist[0].upper() == 'LP':
        umode = 1                   # パルス幅単位 クロック数
    else:
        umode = 0                   # パルス幅単位 μs
    try:
        velof = float(cdatalist[1]) # 合成変化速度
    except ValueError:
        return ret                  # データ異常
    global finterclk                # クロック周波数(MHz)
    if umode == 1:
        velof = velof / finterclk   # 合成変化速度を 単位 μs に

    chnolist = [0,0,0,0,0,0,0,0,0,0,0,0]
    chpwlist = [1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]

    for cnt in range(cdatanum):     # 対象チャンネル数分の繰返し
        chandata = cdatalist[cnt+2].split(',')    # ',' で分離
        if len(chandata) != 2:
            return ret              # データ異常
        try:
            chnolist[cnt] = int(chandata[0])    # チャンネル番号
            chpwlist[cnt] = float(chandata[1])  # パルス幅
            if umode == 1:
                chpwlist[cnt] = chpwlist[cnt] / finterclk  # 単位 μs
        except ValueError:
            return ret              # データ異常

    readdata = Drsw_read(handle, '12')  # パルス幅現在値読取り
    pm = [1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]
    pv = [1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0]
    if len(readdata) == 59:
        p = readdata.split(',')         # ',' で分離
        q = 0.0
        for cnt in range(cdatanum):     # パルス幅移動量算出
            pm[cnt] = chpwlist[cnt] - float(int(p[chnolist[cnt]],16)) / finterclk
            q = q + pm[cnt] **2
        u = math.sqrt(q)                # 合成移動量
        if u < 0.01:
            u = 0.01
        for cnt in range(cdatanum):     # 各チャンネルの変化速度
            pv[cnt] = velof * pm[cnt] / u
            if pv[cnt] > 1000000:
                pv[cnt] = 1000000

        tdata = ''
        for cnt in range(cdatanum):     # 速度設定文字列作成
            tdata = tdata + 'V,' + str(chnolist[cnt]) + ',' + format(pv[cnt],'.2f') + '/'
        tdata = tdata[:-1]              # 末尾の'/'文字削除
        readdata = Drsw_write(handle, tdata)   # 速度設定送信
        if readdata == 'NG':
            return ret                  # データ送受信異常

        tdata = ''
        for cnt in range(cdatanum):     # 目標パルス幅文字列作成
            tdata = tdata + 'W,' + str(chnolist[cnt]) + ',' + format(chpwlist[cnt],'.2f') + '/'
        tdata = tdata[:-1]              # 末尾の'/'文字を削除
        readdata = Drsw_write(handle, tdata)   # 目標パルス幅設定送信
        if readdata == 'NG':
            return ret                  # データ送受信異常
        ret = 'OK'                      # 正常終了とする
    return ret

def Drsw_close(handle):
# 関数 デバイスをCLOSE
#    引数   : デバイスハンドル
#    戻り値 : なし

    FT_rpi.close_dacs(handle)
