# DACS-2500KB カウンタ

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# Dcnt_rpi.py と FT_rpi.py を同じディレクトリに格納

import FT_rpi
import time

direslist = [1,1,1,1,1,1]     # 0:リセット入力無効  1:リセット入力有効
cmodelist = [0,0,0,0,0,0]     # カウントモード

def Dcnt_init(devnum):
# 関数 デバイスのOPEN
#    引数   : デバイス番号(int型)
#    戻り値 : (1)'OK' 正常終了 'NG' 異常 ,(2)デバイスハンドル

    ret = 'NG'            # 仮に異常終了とする
    for cnt in range(4):  # 4デバイス分の検索
        if devnum >= 9:
            devn = cnt
        else:             # デバイス番号を指定している
            devn = devnum
        handle = FT_rpi.init_dacs(devn)
        if handle.value != None:
            # デジタル入力を実行
            readdata = FT_rpi.transfer_dacs(handle, 'W0Rxxxxx'+chr(0xd), 9)
            if (len(readdata) == 9) and (readdata[0:1] == 'R'):
                # カウンタ値を取得してカウンタデバイスの確認
                readdata = FT_rpi.transfer_dacs(handle, 'm00'+chr(0xd), 9)
                if (len(readdata) == 9) and (readdata[0:1] == 'n'):
                    ret = 'OK'
                readdata = FT_rpi.transfer_dacs(handle, 'W0Rxxxxx'+chr(0xd), 9)
                if ret == 'OK':
                    break
            FT_rpi.close_dacs(handle)   # デバイスをCLOSE
        if devnum < 9:
            break
    return ret,handle

def Dcnt_read(handle, chno):
# 関数 カウンタ値の取得
#    引数   : デバイスハンドル, チャンネル番号 文字列 x または xH
#                    x:  カウント値 チャンネル0～5, 9は全ch指定
#                    xH: ホールド値
#                    D:  デジタル入力
#    戻り値 : 文字列 カウント値
#                    全ch指定のときはch0からch5をカンマ区切り 
#                    デジタル入力のときは16進数6桁(bit23--0)
    sdata = ''
    c1 = chno[0:1]            # チャンネル番号

    # 単一チャンネル指定のとき
    if c1 >= '0' and c1 < '6':
        a = int(c1)           # チャンネル番号(int)
        comchar = 'M0'
        if a >= 3:            # 拡張カウンタ
            a = a - 3
            comchar = 'm0'
        if len(chno) == 2:
            if chno[1:2].upper() == 'H':    # ホールド値指定
                a = a + 3
        dcommand = comchar + format(a*2, 'x') + '&'
        dcommand = dcommand + comchar + format(a*2+1, 'x') + chr(0xd)
        # コマンド送信とカウント値受信
        rnum = 18
        sdata = Dcnt_cntread(handle, dcommand, rnum)

    # 全チャンネル指定のとき
    elif c1 == '9':
        dcommand = 'M00&M01&M02&M03&M04&M05&m00&m01&m02&m03&m04&m05'+ chr(0xd)
        if len(chno) == 2:
            if chno[1:2].upper() == 'H':    # ホールド値指定
                dcommand = 'M06&M07&M08&M09&M0A&M0B&m06&m07&m08&m09&m0A&m0B'+ chr(0xd)
        # コマンド送信とカウント値受信
        rnum = 108
        sdata = Dcnt_cntread(handle, dcommand, rnum)

    # デジタル入力指定のとき
    elif c1 == 'D':
        tdata = 'W0R' + chr(0xd)
        readdata = FT_rpi.transfer_dacs(handle, tdata, 9)
        if len(readdata) == 9:
            if readdata[0:1] == 'R':
                sdata = readdata[2:8]
    return sdata

def Dcnt_write(handle, wdata):
# 関数 コマンド送信とレスポンス受信
#    引数   : デバイスハンドル, 指示データ
#    戻り値 : 'OK' 正常終了 'NG' 異常終了
#       データ連結では、一部のデータが不正のとき、すべての転送なし

# 指示データ(文字列)
# /(スラッシュ)で複数の指示データを連結可能  例 0R/0G/1R/1G
#          x:チャンネル番号(0～5)
#     xG  カウント開始
#     xS  カウント停止(電源投入初期)
#     xR  カウント値リセット
#     xF,zzz  フィルタ設定   zzz 小数点可 単位μs
#     xAB A相/B相カウント
#     xUD up/downカウント(電源投入初期)
#     xR1 カウントリセット入力有効(電源投入初期)
#     xR0 カウントリセット入力無効(例 ch0ではDI02)
#     xT1 ゲート入力有効
#     xT0 ゲート入力無効(電源投入初期)
#     xMI 周期計測
#     xMW パルス幅計測
#     xMN 通常カウント動作(電源投入初期)
#     xEC,z--z  最終値設定(10進数)  くり返し動作
#     xES,z--z  最終値設定(10進数)  最終値で停止
#     xEN 最終値設定解除(電源投入初期)
#     DI,y--y デジタル入力極性(16進数6桁 bit23～0)
#     DL1 デジタル出力(bit11～0)  カウンタ用出力に変更
#     DL0 デジタル出力(bit11～0)  汎用出力用途(初期)
#     DH1 デジタル出力(bit23～16) カウンタ用出力(カウンタ使用時)
#     DH0 デジタル出力(bit23～16) 汎用出力用途に変更
#     DO,y--y デジタル出力(16進数6桁 bit23～0)

    global direslist                # 0:リセット入力無効  1:リセット入力有効
    global cmodelist                # カウントモード

    ret = 'NG'                      # 仮に異常終了とする
    cdatalist = wdata.split('/')    # '/' で分離
    alltdata = ''                   # コマンド文字列の準備

    for cntlist in range(len(cdatalist)):  # 指示データ数分の繰返し
        cdata = cdatalist[cntlist].split(',')     # ',' で分離
        clen = len(cdata)           # ','区切りのデータ数
        c0len = len(cdata[0])
        if c0len > 1:
            ca = (cdata[0]).upper()
            c1 = ca[0:1]            # 最初の1文字
        else:
            alltdata = ''
            break

        # コマンド文字列に変換

        tdata = ''                  # 1指示データ分のコマンド文字列の準備

        if c1 == 'D':
            if clen == 2 and ca == 'DI':   # デジタル入力極性設定
                if len(cdata[1]) == 6:
                    tdata = 'Y0'+ cdata[1] + '&'
            elif ca == 'DL1':       # デジタル出力(bit11～0)用途設定
                tdata = 'm0E&'      # カウンタ用出力
            elif ca == 'DL0':
                tdata = 'm0F&'      # 汎用出力
            elif ca == 'DH1':       # デジタル出力(bit23～16)用途設定
                tdata = 'M0E&'      # カウンタ用出力
            elif ca == 'DH0':
                tdata = 'M0F&'      # 汎用出力
            elif clen == 2 and ca == 'DO': # デジタル出力(bit23～0)設定
                if len(cdata[1]) == 6:
                    tdata = 'W0' + cdata[1] + '&'
            if len(tdata) != 0:
                alltdata = alltdata + tdata
                continue
            else:
                alltdata = ''
                break

        if (c1 < '0') or (c1 > '5'):  # チャンネル番号の確認
            alltdata = ''
            break
        cno = int(c1)               # チャンネル番号をintに変換
        if cno > 2:
            comcharl = 'm0' + format((cno-3)*2,'X')   # 拡張カウンタ low word
            comcharh = 'm0' + format((cno-3)*2+1,'X') # 拡張カウンタ high word
        else:
            comcharl = 'M0' + format(cno*2,'X')       # low word
            comcharh = 'M0' + format(cno*2+1,'X')     # high word
        c2 = ca[1:2]                # 2文字目

        if c0len == 2:
            # カウント開始
            if c2 == 'G':
                if direslist[cno] == 0:     # リセット入力無効のとき
                    tdata = comcharl + 'A&'
                else:
                    tdata = comcharl + '8&'
            # カウント停止
            elif c2 == 'S':
                if direslist[cno] == 0:     # リセット入力無効のとき
                    tdata = comcharl + '6&'
                else:
                    tdata = comcharl + '4&'
            # カウント値リセット
            elif c2 == 'R':
                if direslist[cno] == 0:     # リセット入力無効のとき
                    tdata = comcharl + '3&'
                else:
                    tdata = comcharl + '1&'

            # フィルタ設定
            elif clen == 2 and c2 == 'F':
                try:
                    cf = float(cdata[1])    # 単位μs
                    cf = cf/0.015625        # 最小単位 15.625ns
                    for cnt in range(5):
                        if cf < 30000.0:
                            break;
                        cf = cf/2
                    cnt = 4 - cnt           # 時間単位
                    if cf > 65500.0:
                        cf = 65500.0        # 最大値を制限
                    cdata = int(cf)
                    if cdata > 0:
                        sdata = format(cdata -1, '04X')
                except ValueError:
                    alltdata = ''
                    break
                if cdata > 0:               # フィルタ設定
                    tdata = 'T0' + format(cnt+8,'X') + format(cno*2,'X') + sdata + '&'
                else:                       # フィルタ解除
                    tdata = 'T00' + format(cno*2,'X') + '&'

        elif c0len == 3:
            c3 = ca[2:3]                    # 3文字目

            if (c2 == 'U' and c3 == 'D') or (c2 == 'A' and c3 == 'B'):
            # up/downカウント
                if c2 == 'U':
                    cmodelist[cno] = cmodelist[cno] & 0x7
            # A相/B相カウント
                else:
                    cmodelist[cno] = cmodelist[cno] | 0x8
                tdata = comcharh + format(cmodelist[cno], 'X') + '&'
            # カウントリセット入力有効/無効
            elif c2 == 'R' and (c3 == '0' or c3 == '1'):
                if c3 == '0':           # リセット入力無効
                    direslist[cno] = 0
                    tdata = comcharl + '2&'
                else:                   # リセット入力有効
                    direslist[cno] = 1
                    tdata = comcharl + '0&'
            # ゲート入力有効/無効
            elif c2 == 'T' and (c3 == '0' or c3 == '1'):
                if c3 == '1':           # ゲート入力有効
                    cmodelist[cno] = cmodelist[cno] | 0x2
                else:                   # ゲート入力無効
                    cmodelist[cno] = cmodelist[cno] & 0xd
                tdata = comcharh + format(cmodelist[cno], 'X') + '&'
            elif c2 == 'M' and (c3 == 'N' or c3 == 'I' or c3 == 'W'):
                cmodelist[cno] = cmodelist[cno] & 0x9  # 通常カウント動作
            # 周期計測
                if c3 == 'I':
                    cmodelist[cno] = cmodelist[cno] | 0x4
            # パルス幅計測
                elif c3 == 'W':
                    cmodelist[cno] = cmodelist[cno] | 0x6
                tdata = comcharh + format(cmodelist[cno], 'X') + '&'

            # 最終値設定/解除
            elif c2 == 'E':
                if c3 == 'N':
                    cmodelist[cno] = cmodelist[cno] & 0xe
                    if direslist[cno] == 0:     # リセット入力無効のとき
                        tdata = comcharl + '2FFFF&'
                    else:
                        tdata = comcharl + '0FFFF&'
                    tdata = tdata + comcharh + format(cmodelist[cno], 'X') + 'FFFF&'
                elif clen == 2 and (c3 == 'C' or c3 == 'S'):
                    try:
                        cdata = int(cdata[1])
                        sdata = format(cdata, '08X')
                    except ValueError:
                        alltdata = ''
                        break
                    if direslist[cno] == 0:     # リセット入力無効のとき
                        tdata = comcharl + '2'
                    else:
                        tdata = comcharl + '0'
                    if c3 == 'S':
                        cmodelist[cno] = cmodelist[cno] | 0x1
                    else:
                        cmodelist[cno] = cmodelist[cno] & 0xe
                    tdata = tdata + sdata[4:8] + '&'
                    tdata = tdata + comcharh + format(cmodelist[cno], 'X') + sdata[0:4] + '&'

        if len(tdata) != 0:
            alltdata = alltdata + tdata     # コマンド文字列の連結
        else:
            alltdata = ''
            break

    # 正常な入力データのときコマンドを送信
    if len(alltdata) != 0:
        num = alltdata.count('&')       # コマンド数
        alltdata = alltdata[:-1] + chr(0xd)  # 末尾の'&'文字をCRに置換
        readdata = FT_rpi.transfer_dacs(handle, alltdata, 0)
        if len(readdata) == num*9:
            ret = 'OK'                  # 正常終了とする
    return ret

def Dcnt_close(handle):
# 関数 デバイスをCLOSE
#    引数   : デバイスハンドル
#    戻り値 : なし

    FT_rpi.close_dacs(handle)

def Dcnt_cntread(handle, dcommand, rnum):
# 関数 カウンタ値読取りコマンド送信/受信
#     引数   : デバイスハンドル, 
#              カウンタ値読取りコマンド文字列、文字数(18または108)
#     戻り値 : カウンタ値(10進数文字列 全チャンネルの場合はカンマ区切り)

    sdata = ''
    cntlast = 0
    # コマンド送信とカウント値受信
    readdata = FT_rpi.transfer_dacs(handle, dcommand, rnum)  # (9文字)X(コマンド数)
    # 正常にデータを受信したとき
    if len(readdata) == 108:
        if readdata[0:1] == 'N' and readdata[54:55] == 'n':
            cntlast = 6
    elif len(readdata) == 18:
        cntlast = 1
    if cntlast != 0:
        for cnt in range(cntlast):
            # 16進数4桁文字列の上位wordと下位wordを連結
            p = cnt*18 + 4
            c3 = readdata[(p+9):(p+13)] + readdata[(p):(p+4)]
            try:
                cdata = int(c3,16)
                if cdata & 0x80000000:                  # 負数の処理
                    cdata = cdata - 0x100000000
                if cnt != 0:
                    sdata = sdata + ','
                sdata = sdata + ("{:d}".format(cdata))  # 文字列に変換
            except ValueError:
                sdata = ''
                break;
    # エラーのとき、コマンドを送信し、残留データを読み切る
    else:
        time.sleep(1)
        readdata = FT_rpi.transfer_dacs(handle, 'W0000000'+chr(0xd), 0)
    return sdata
