# DACS-2500KB-SCN3 カウンタ サンプルプログラム

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# Dcnt_rpi_sample.py、Dcnt_rpi.py、FT_rpi.py を同じディレクトリに格納

# デバイスをUSBポートに接続
# Dcnt_rpi_sample.py を起動すると
# カウント値読取りコマンドを自動送信し、
# 6チャンネル分のカウント値を読取って連続表示します。
# ホールド値読取りコマンドを自動送信し、
# 6チャンネル分のホールド値を読取って連続表示します。

# マウスのセンターキー(スクロールキー)を押すとキー入力に移行します。

# 以下のキー入力後、enterにて指示データを送信
# /(スラッシュ)で複数の指示データを連結可能  例 0R/0G/1R/1G
#          x:チャンネル番号(0～5)
#     xG  カウント開始
#     xS  カウント停止(電源投入初期)
#     xR  カウント値リセット
#     xF,zzz  フィルタ設定   zzz 小数点可 単位μs
#     xAB A相/B相カウント
#     xUD up/downカウント(電源投入初期)
#     xR1 カウントリセット入力有効(電源投入初期)
#     xR0 カウントリセット入力無効(例 ch0ではDI02)
#     xT1 ゲート入力有効
#     xT0 ゲート入力無効(電源投入初期)
#     xMI 周期計測
#     xMW パルス幅計測
#     xMN 通常カウント動作(電源投入初期)
#     xEC,z--z  最終値設定(10進数)  くり返し動作
#     xES,z--z  最終値設定(10進数)  最終値で停止
#     xEN 最終値設定解除(電源投入初期)
#     DI,y--y デジタル入力極性(16進数6桁 bit23～0)
#     DL1 デジタル出力(bit11～0)  カウンタ用出力に変更
#     DL0 デジタル出力(bit11～0)  汎用出力用途(初期)
#     DH1 デジタル出力(bit23～16) カウンタ用出力(カウンタ使用時)
#     DH0 デジタル出力(bit23～16) 汎用出力用途に変更
#     DO,y--y デジタル出力(16進数6桁 bit23～0)

# enterのみ入力にてカウンタ値表示を再開
# -(マイナス) enter にてプログラム終了

import Dcnt_rpi
import time

#******** マウスのセンターキー(スクロールキー)検知処理 ********
import threading
import evdev      # マウスのセンターキー検出に使用しています。
                  # インストールが必要  例 pip3 install evdev
                  # DACS販売型式 RPi-4B2G には evdev インストール済
kbdevice = evdev.InputDevice('/dev/input/event0')
                  # キーボード入力に移行しない場合は、
                  # event0 を、マウスに割当てられた番号に変更してください。
                  # 例 event0 event1 event2 event3 など
kbwaitf = 0       # キーボードから入力中  0:入力中ではない, 1:入力中

# マウスのセンターキーを押したときを検出
def kbhit_read():
    global kbwaitf                          # キーボード入力中をグローバル変数とする
    kbwaitf = 0                             # 仮に入力中でないとする
    for event in kbdevice.read_loop():
        if event.type == evdev.ecodes.EV_KEY:
            if event.value == 1:            # キーを押したとき
                if event.code == evdev.ecodes.BTN_MIDDLE:   # センターキーを押した
                    kbwaitf = 1             # キーボード入力中とする

# キーボードから指示データ入力
def kbinput_read():
    t0 = time.time()
    ret = ''
    while True:
        kybuf = input('keyin = ')
        if (time.time() - t0) > 0.3:        # 新しく入力したデータ
            ret = kybuf
            break
        else:                               # 入力開始以前のデータを破棄
            print('\033[1A')                # 表示ラインを戻す
    return ret

# マウスのセンターキー検出 threadを開始
th = threading.Thread(target=kbhit_read, daemon=True)
th.start()

#******** ここからが カウンタ制御の本題です ********

# デバイスをOPEN、カウンタボードを確認
ret = Dcnt_rpi.Dcnt_init(9)
if ret[0] == 'OK':
    handle = ret[1]     # デバイスハンドルをセット
else:
    print('no device')
    time.sleep(3)
    exit()

disp1 = ' |   ch0     |     1     |     2     |     3     |     4     |     5     |    '

# カウント値またはホールド値の画面表示を繰返す
while True:
    time.sleep(0.01)

    print(disp1)        # カウンタ番号0～5を画面表示
    for lcnt in range(2):
        if lcnt == 0:   # 6チャンネル分のカウント値を取得
            readdata = Dcnt_rpi.Dcnt_read(handle, '9')
            sdata = 'C'
        else:           # 6チャンネル分のホールド値を取得
            readdata = Dcnt_rpi.Dcnt_read(handle, '9H')
            sdata = 'H'
        if len(readdata) != 0:           # 正常データのとき
            pdata = readdata.split(',')  # ',' で分離
            for cnt in range(6):         # 1行に連結
                sdata = sdata + ' '*(12-len(pdata[cnt])) + pdata[cnt]
        else:                            # 不正データのとき
            sdata = 'data error'+' '*64
            time.sleep(2)
        print(sdata + '    ')            # 6チャンネル分を表示
    if kbwaitf == 0:     # キーボードからの入力中でないとき
        print('\033[4A')                 # 表示ラインを4行戻す
        continue

# キーボードから指示データを入力
    kydatabuf = kbinput_read()
    kbwaitf = 0                          # キー入力中を解除
    # マイナスキー入力のときは終了
    if kydatabuf == '-':
        break
    if len(kydatabuf) != 0:
        # 指示データ送信
        readdata = Dcnt_rpi.Dcnt_write(handle, kydatabuf)
        # 応答結果を表示
        if readdata == 'OK':
            print(kydatabuf+'-->'+readdata)
        else:
            print('\033[31m'+kydatabuf+'-->'+readdata+'\033[0m')

# デバイスをCLOSE
Dcnt_rpi.Dcnt_close(handle)
