# DACS-2500K(KB) モーションコントローラ サンプルプログラム

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# Dpmv_rpi_sample.py、Dpmv_rpi.py、FT_rpi.py を同じディレクトリに格納

# デバイスをUSBポートに接続
# Dpmv_rpi_sample.py を起動すると
# 各軸移動量を初期化します。
#    電源投入時に、デバイス側でも初期化をしますが、
#    このプログラムでの初期化処理は、
#        デバイスを再接続しないで、
#        プログラムを再起動した場合の対策です。

# 位置読取りコマンドを自動送信し、
#    6軸分の位置を読取って連続表示します。

# マウスのセンターキー(スクロールキー)を押すとキー入力に移行します。

# M enter とキー入力すると
#    移動量表示に変わります。
# P enter とキー入力すると
#    位置表示に戻ります。
# 表示を一時停止するときは PまたはMを押す
#    enter にて表示再開

# 以下のキー入力後、enterにて指示データを送信
# /(スラッシュ)で複数の指示データを連結可能 例 1,100/2,-1000
# x,y--y 各軸の移動量を指定
#      x:軸番号(1～6)  y--y:移動量(単位 パルス)
#      x=9 のとき、全軸に同じ移動量をセット
# V,y--y 移動速度を指定(単位 Hz)
# A,y--y 加速度を指定(単位 KHz/s)
# G 移動開始
# S 強制停止
# その他の操作詳細は、Dpmv_rpi.py 関数の注釈をご覧ください。

# enterのみ入力にて表示を再開
# -(マイナス) enter にてプログラム終了

import Dpmv_rpi
import time

#******** マウスのセンターキー(スクロールキー)検知処理 ********
import threading
import evdev      # マウスのセンターキー検出に使用しています。
                  # インストールが必要  例 pip3 install evdev
                  # DACS販売型式 RPi-4B2G には evdev インストール済
kbdevice = evdev.InputDevice('/dev/input/event0')
                  # キーボード入力に移行しない場合は、
                  # event0 を、マウスに割当てられた番号に変更してください。
                  # 例 event0 event1 event2 event3 など
kbwaitf = 0       # キーボードから入力中  0:入力中ではない, 1:入力中

# マウスのセンターキーを押したときを検出
def kbhit_read():
    global kbwaitf                          # キーボード入力中をグローバル変数とする
    kbwaitf = 0                             # 仮に入力中でないとする
    for event in kbdevice.read_loop():
        if event.type == evdev.ecodes.EV_KEY:
            if event.value == 1:            # キーを押したとき
                if event.code == evdev.ecodes.BTN_MIDDLE:   # センターキーを押した
                    kbwaitf = 1             # キーボード入力中とする

# キーボードから指示データ入力
def kbinput_read():
    t0 = time.time()
    ret = ''
    while True:
        kybuf = input('keyin = ')
        if (time.time() - t0) > 0.3:        # 新しく入力したデータ
            ret = kybuf
            break
        else:                               # 入力開始以前のデータを破棄
            print('\033[1A')                # 表示ラインを戻す
    return ret

# マウスのセンターキー検出 threadを開始
th = threading.Thread(target=kbhit_read, daemon=True)
th.start()

#******** ここからが モーションコントローラの本題です ********

# デバイスをOPEN、各軸移動量を初期化
ret = Dpmv_rpi.Dpmv_init(9)
if ret[0] == 'OK':
    handle = ret[1]     # デバイスハンドルをセット
    print('OK')
else:
    print('no device')
    time.sleep(3)
    exit()

dispmode = 'P'              # 表示内容選択  M:移動量表示  P:位置表示
disp1 = '|  m.1  |  m.2  |  m.3  |  m.4  |  m.5  |  m.6  |  DI   |A 6 state|    '
disp2 = '|  p.1  |  p.2  |  p.3  |  p.4  |  p.5  |  p.6  |  DI   |A 6 state|    '

# | m.1～6| 各軸移動量  | p.1～6| 各軸位置
# |  DI   |  デジタル入力 bit23～0を16進6桁表示
# |A 6 state|
#  A:全軸の移動状態 6:第6軸独立動作状態 state:ステータス(16進下位5桁表示)

print(disp2)                # 軸番号などを画面表示

# 移動量または位置、デジタル入力、ステータスの画面表示を繰返す
while True:
    time.sleep(0.01)

    # 移動量または位置とデジタル入力を取得
    readdata = Dpmv_rpi.Dpmv_read(handle, dispmode)
    # ステータスを取得
    sreaddata = Dpmv_rpi.Dpmv_sread(handle)

    # 1軸から6軸までの移動量または位置とデジタル入力、ステータスを表示
    if len(readdata) != 0 and len(sreaddata) != 0:  # 正常データのとき
        pdata = readdata.split(',')      # ',' で分離
        sdata = ''
        for cnt in range(7):       # 1行に連結
            sdata = sdata + ' '*(8-len(pdata[cnt])) + pdata[cnt]
        print(sdata + ' ' + sreaddata + '    ',end='\r')
    else:                          # 不正データのとき
        print('data error'+' '*54)
        time.sleep(2)
    if kbwaitf == 0:     # キーボードからの入力中でないとき
        continue

# キーボードから指示データを入力
    print()
    kydatabuf = kbinput_read()
    kbwaitf = 0                    # キー入力中を解除
    # マイナスキー入力のときは終了
    if kydatabuf == '-':
        break
    if len(kydatabuf) != 0:
        # 先頭がMまたはPのときは、表示内容セット
        c1 = kydatabuf.upper()
        if c1 == 'M' or c1 == 'P':
            dispmode = c1              # 表示内容セット
        # 指示データ送信
        else:
            if kydatabuf[0:1].upper() == 'M':   # MOVE MDまたはMI 送信
                if kydatabuf[2:3] == '6':       # 6軸単独動作
                    readdata = Dpmv_rpi.Dpmv_move6(handle, kydatabuf)
                else:                           # 全軸動作
                    readdata = Dpmv_rpi.Dpmv_moveA(handle, kydatabuf)
            else:                               # コマンド送信
                readdata = Dpmv_rpi.Dpmv_write(handle, kydatabuf)
            # 応答結果を表示
            if readdata[0:2] == 'OK':
                print(kydatabuf+'-->'+readdata)
            else:
                if readdata == 'ER':
                    ms = 'connection error'
                elif readdata == 'NA':
                    ms = 'moving error'
                else:
                    ms = 'data error'
                print('\033[31m'+kydatabuf+'-->' + ms + '\033[0m')
    # 移動量または位置の軸番号などを表示
    if dispmode == 'M':
        print(disp1)              # 移動量の軸番号を表示
    else:
        print(disp2)              # 位置の軸番号を表示

# デバイスをCLOSE
Dpmv_rpi.Dpmv_close(handle)
