# DACS-2500KB-PM63 モーションコントローラ サンプルプログラム

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# d15pmv_rpi.py と FT_rpi.py を同じディレクトリに格納

# デバイスをUSBポートに接続
# d15pmv_rpi.py を起動すると

# 各軸移動量、速度と加速度、位置を初期化します。
#    電源投入時に、デバイス側でも初期化をしますが、
#    このプログラムでの初期化処理は、
#        デバイスを再接続しないで、
#        プログラムを再起動した場合の対策です。

# 移動量読取りコマンドを自動送信し、
#    6軸分の移動量を読取って連続表示します。

# キーボードのenterキーのみを押すとキー入力に移行します。

# P enter とキー入力すると
#    位置表示に変わります。
# M enter とキー入力すると
#    移動量表示に戻ります。

# x,y--y enter とキー入力にて各軸の移動量を指定
#      x:軸番号(0～5)  y--y:移動量(単位 パルス)
# v,y--y enter にて移動速度を指定(単位 0.25Hz)
# a,y--y enter にて加速度を指定(単位 1.25KHz/s)
# g enter にて移動開始
# T enter にて強制停止

# W0000000(enter)などのコマンド文字列入力にて
#    デバイスを操作することができます。

# enterのみ入力にて表示を再開
# マイナスキーとenterを入力にてプログラム終了

import FT_rpi
import time

#******** キーボード入力検知処理 ********
import threading
import evdev      # キーボードからのキー入力検出に使用しています。
                  # インストールが必要  例 pip3 install evdev
                  # DACS販売型式 RPi-4B2G には evdev インストール済
kbdevice = evdev.InputDevice('/dev/input/event0')
                  # キーボードからのキー入力検出に使用しています。
                  # キーボードからの入力に移行しない場合は、
                  # event0 を、キーボードに割当てられた番号に変更してください。
                  # 例 event0 event1 event2 event3 など
kbwaitf = 0       # キーボードから入力中  0:入力中ではない, 2:enterキーを押した

# キーボードのenterキーを押したときを検出
def kbhit_read():
    global kbwaitf                              # キーボード入力中をグローバル変数とする
    kbwaitf = 0                                 # 仮に入力中でないとする
    for event in kbdevice.read_loop():
        if event.type == evdev.ecodes.EV_KEY:
            if event.value == 1:                # キーを押したとき
                if event.code == 28:
                    kbwaitf = 2                 # enterキーを押したとき

# キーボードからのキー入力検出 threadを開始
th = threading.Thread(target=kbhit_read, daemon=True)
th.start()

#******** ここからが モーションコントローラの本題です ********

# デバイスをOPEN
handle = FT_rpi.init_dacs(0)
if handle.value == None:
    print('no device')
    exit()

# 各軸移動量を初期化
readdata = ''
for cnt in range(6):
    readdata = readdata + FT_rpi.transfer_dacs(handle, 'P0' + format(cnt,'x') +'00000'+chr(0xd), 9)
# 各軸移動量保存。軸移動開始のマスター軸指定用
axmvlist = [0,0,0,0,0,0]
# 速度と加速度を初期化
readdata = readdata + FT_rpi.transfer_dacs(handle, 'P0800001'+chr(0xd), 9)
readdata = readdata + FT_rpi.transfer_dacs(handle, 'P0900010'+chr(0xd), 9)
# 位置/パルス分配異常リセット
readdata = readdata + FT_rpi.transfer_dacs(handle, 'Q0A'+chr(0xd), 9)
readdata = readdata + FT_rpi.transfer_dacs(handle, 'Q0B'+chr(0xd), 9)

if len(readdata) != 90:   # 10コマンド分の受信データ数を確認
    print('data error')
    exit()

# 移動量読取りコマンド文字列を準備
c1command = 'Q00&Q01&Q02&Q03&Q04&Q05&Q06&W0R' + chr(0xd)
# 位置読取りコマンド文字列を準備
c2command = 'q00&q01&q02&q03&q04&q05&Q06&W0R' + chr(0xd)

dispmode = 0                # 表示内容選択  0:disp1  1:disp2
dcommand = c1command        # 送信要求コマンド文字列をdisp1用とする

disp1 = '|  m.0  |  m.1  |  m.2  |  m.3  |  m.4  |  m.5  |status |'
disp2 = '|  p.0  |  p.1  |  p.2  |  p.3  |  p.4  |  p.5  |status |'
print(disp1)                # 軸番号を画面表示

c1 = ' '         # 応答1番目の文字

# コマンドコードのリスト(キー入力チェック用)
comlist = ['W','P','Q','q','I','Z']

# 移動量または位置の取得と画面表示を繰返す
while True:
    time.sleep(0.01)

    # 移動量または位置とデジタル入力を取得。受信文字数：(9文字)X(8コマンド)=72
    readdata = FT_rpi.transfer_dacs(handle, dcommand, 72)

    if len(readdata) != 72 or (readdata[0] != 'S' and readdata[0] != 's'):
        # エラーのとき、コマンドを送信し、残留データを読み切る
        time.sleep(1)
        readdata = FT_rpi.transfer_dacs(handle, 'W0R'+chr(0xd), 0)
        if len(readdata) < 9: # それでも応答がないとき
            print("")
            print('time-out error')
            break;          # ループを終了
        else:
            continue;       # 応答があればループを継続

    # 正常にデータを受信したとき
    sdata = ''              # 画面表示用の文字列を初期化

    for cnt in range(7):
        p = cnt*9 + 3       # 移動量または位置の文字列中位置

        # 16進数5桁文字列を10進数の整数に変換
        try:
            cdata = int(readdata[(p):(p+5)],16)
        except ValueError:
            break

        if cnt != 6:        # 移動量または位置のとき
            # 負方向の処理
            if (cdata & 0x80000) != 0:
                if dispmode == 0:
                    # 移動量の場合: bit19が符号で以下は絶対値
                    cdata = 0x80000 - cdata
                else:
                    # 位置の場合  : 20bitの純2進数
                    cdata = cdata - 0x100000

            # 画面表示用に8桁の文字列で連結する
            sdata = sdata + ("{:8d}".format(cdata))
        else:               # ステータスのとき
            cdata = cdata & 0xfff
            sdata = sdata + '  ' + ("{:03x}".format(cdata)) + '(H)  '

    # 移動量または位置を画面表示
    if kbwaitf != 2:     # キーボードからの入力中でないとき
        if len(sdata) == 58:    # 正常データのとき
            print('\r' + sdata,end='\r')
        else:                   # 不正データのとき
            print('data error'+' '*46)
            time.sleep(2)
        continue
    # enterキーを押してないときは、ここまでの画面表示を繰り返す

    # キーボードのenterキーを押したとき
    kydatabuf = input()    # 最初のenterキーまでを無効データとしてを読み捨てる

    # 以下のキー入力処理を繰り返す

    while True:
        kydatabuf = input('\r' + 'keyin = ')

        # キー入力中にenterキーのみ入力のとき
        if len(kydatabuf) == 0:
            kbwaitf = 0        # キー入力中を解除
            if dispmode == 0:  # 軸番号を再表示して連続表示に戻る
                print('\r' + disp1)
            else:
                print('\r' + disp2)
            break

        # 最初の1文字
        c1 = kydatabuf[0]
        if c1 == '-':          # マイナスキーのときはプログラム終了へ
            break

        # コマンド文字列の場合
        if c1 in comlist and len(kydatabuf) > 1:
            # キー入力データを送信し応答を受信
            tdata = kydatabuf+chr(0xd)
            readdata = FT_rpi.transfer_dacs(handle, tdata, 0)
            print('\r' + kydatabuf + ' --> ' + readdata)     # 応答を画面表示
            continue

        # 先頭がMまたはP
        if c1 == 'M' or c1 == 'm':
            dispmode = 0
            dcommand = c1command       # 送信要求文字列をdisp1用に
            continue
        elif c1 == 'P' or c1 == 'p':
            dispmode = 1
            dcommand = c2command       # 送信要求文字列をdisp2用に
            continue

        kydata = kydatabuf.split(',')  # ',' で分離

        # キー入力をコマンド文字列に変換するとき
        tdata = ''                     # コマンド文字列の準備
        # 先頭が0～5の軸番号指定
        if c1 >= '0' and c1 < '6':
            axis = int(c1,10)          # 軸番号を整数に変換
            try:
                a = int(kydata[1],10)  # intに変換
                b = abs(a)             # 絶対値
                axmvlist[axis] = b     # 移動量(絶対値)を保存
                if b > 0x7ffff:        # 最大値の制限
                    b = 0x7ffff
                if a < 0:              # 負数のとき
                    b = b + 0x80000    # bit20 に負の符号
                tdata = format(b, '05x')  # 16進数5桁に変換

                # 移動量指定コマンドを準備
                tdata = 'P0' + c1 + tdata + chr(0xd)
            except:
                tdata = ''             # 不正データのとき

        # 速度指定
        elif c1 == 'V' or c1 == 'v':
            try:
                a = int(kydata[1],10)  # intに変換
                if a > 0xf4240:        # 最大値の制限
                    a = 0xf4240
                if a < 1:              # 最小値の制限
                    a = 1
                tdata = format(a, '05x')  # 16進数5桁に変換

                # 速度指定コマンドを準備
                tdata = 'P08' + tdata + chr(0xd)
            except:
                tdata = ''             # 不正データのとき

            # 加速度指定
        elif c1 == 'A' or c1 == 'a':
            try:
                a = int(kydata[1],10)  # intに変換
                if a > 0xfff:          # 最大値の制限
                    a = 0xfff
                if a < 1:              # 最小値の制限
                    a = 1
                tdata = format(a, '05x')  # 16進数5桁に変換
                # 加速度指定コマンドを準備
                tdata = 'P09' + tdata + chr(0xd)
            except:
                tdata = ''             # 不正データのとき

        # 軸移動開始
        elif c1 == 'G' or c1 == 'g':
            # 移動量が最大の軸を判定
            a = 0 
            axis = 0
            for cnt in range(6):
                if axmvlist[cnt] > a:
                    axis = cnt
                    a = axmvlist[cnt]

            # 軸移動開始コマンドを準備
            tdata = 'Q08' + format(axis,'x') + chr(0xd)

        # 軸移動強制停止
        elif c1 == 'T' or c1 == 't':
            # 軸移動停止コマンドを準備
            tdata = 'Q09' + chr(0xd)

        # 位置リセット
        elif c1 == 'R' or c1 == 'r':
            # 位置リセットコマンドを準備
            tdata = 'Q0B' + chr(0xd)

        # 正常な入力データのとき
        if len(tdata) != 0:
            # コマンドを送信
            readdata = FT_rpi.transfer_dacs(handle, tdata, 0)
            # 送信コマンドと応答を画面表示
            print('\r' + tdata[0:-1] + ' --> ' + readdata) 
        # 入力データが不正のとき
        else:
            print('invalid data')

    if c1 == '-':    # 最初の文字がマイナスキーはプログラム終了
        break

# デバイスをCLOSE
FT_rpi.close_dacs(handle)
