# DACS-2500KB-RSV3 PWMパルス出力 サンプルプログラム
#   PWMパルス出力 12チャンネル対象
#   パルス幅変化速度設定を使用した、
#   各チャンネル同時移動開始、同時移動終了の補間制御

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# d15rsv_rpi.py と FT_rpi.py を同じディレクトリに格納

# デバイスをUSBポートに接続
# d15rsv_rpi.py を起動すると
# 以下の初期設定をします。
#    パルス出力の周波数 (内部カウントクロック1MHz/パルス周波数50Hz)
#    全チャンネルのパルス幅 1520us
#    パルス出力の開始

# 動作対象はAグループの12チャンネルです

# 各チャンネルの移動/停止状態を表示
#    例 0000000000   左:ch11--右ch0
#    1: 移動中  0:停止中
# 表示指定チャンネルのパルス幅(現在->目標)を連続表示
#    例 ch 0 1520 -> 3000

# ******** キー操作方法 ********

# 任意チャンネルの目標(次に移動する)パルス幅を指定します。
#  x,y--y(enter)
#            x : 0～11 チャンネル番号
#            y--y : 目標パルス幅
#  x(enter)
#    チャンネル番号のみで、停止状態でのパルス幅を読取ります。
#  x は、パルス幅連続表示の指定チャンネルになります。

# パルス幅変化時間(移動速度)を指定します。
#  V,y--y(enter)
#            V : 文字 V
#            y--y : パルス幅変化時間
#                       指定値ｘ10us で1パルス幅の変化
#            複数チャンネル移動の場合は合成速度となります
#            各チャンネルは、同時スタート、同時停止の補間速度にて移動

# 移動開始
#  G(enter)
#     移動開始後、例 001001xx0011 などと表示  左:ch11--右ch0
#            1: 移動対象チャンネル  0:移動のないチャンネル
#            x: 移動速度が遅すぎて適切な補間速度とならないチャンネル

# 移動一時停止
#  T(enter)
#     停止後、G(enter)にて移動再開
#     停止後に、いずれかチャンネルの目標パルス幅を変更した場合は、
#            補間動作ができなくなるので、
#            G(enter)にて移動再開で、移動対象となる全チャンネルが
#            V で指定したパルス幅変化時間(移動速度)で変化

# W0000000(enter)などのコマンド文字列入力にて
#    デバイスを操作することができます。

# enter のみをキー入力するとプログラムが終了します。

import FT_rpi
import time
import math

#******** キーボード入力検知処理 ********
import threading
import evdev      # キーボードからのキー入力検出に使用しています。
                  # インストールが必要  例 pip3 install evdev
                  # DACS販売型式 RPi-4B2G には evdev インストール済
kbdevice = evdev.InputDevice('/dev/input/event0')
                  # キーボードからのキー入力検出に使用しています。
                  # キーボードからの入力に移行しない場合は、
                  # event0 を、キーボードに割当てられた番号に変更してください。
                  # 例 event0 event1 event2 event3 など
kbwaitf = 0       # キーボードから入力中  0:入力中ではない, 2:キーを押した

# キーボードのキーを押したときを検出
def kbhit_read():
    global kbwaitf                              # キーボード入力中をグローバル変数とする
    for event in kbdevice.read_loop():
        if event.type == evdev.ecodes.EV_KEY:
            if event.value == 1: kbwaitf = 2    # キーを押したとき

# キーボードからのキー入力検出 threadを開始
th = threading.Thread(target=kbhit_read, daemon=True)
th.start()

#******** ここからが PWMパルス出力制御の本題です ********

# パルス幅制限値、中心位置、初期位置を準備
#   Aグループ ch0 --> ch11 の順
pwalow  = [560,560,560,560,560,560,560,560,560,560,560,560]             # 最小値
pwahigh = [2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480] # 最大値
pwalist = [1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520] # 初期位置(その後現在値)
pwanext = [1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520] # 目標パルス幅
pwavel  = 100.0             # パルス幅変化時間(移動速度)
pwamov  = 0                 # 移動中または一時停止中:1  停止中:0
pwanset = 0                 # 次のパルス幅変更 あり:1 なし:0  (一時停止後の補間動作有効／無効の判定に使用)
pwansta = -1                # 各チャンネルの移動状態(-1は無効データ)
pwadisp = 0                 # パルス幅表示チャンネル

# デバイスをOPEN
handle = FT_rpi.init_dacs(0)
if handle.value == None:
    print('no device')
    exit()

# 周波数を初期設定 (内部カウントクロック 1MHz / パルス周波数 50Hz)
#   ch0-->ch11
readdata = FT_rpi.transfer_dacs(handle, 'Q0B04E1F' + chr(0xd), 9)
if len(readdata) != 9:          # 最初の通信でPWMボードの確認
    FT_rpi.close_dacs(handle)   # 受信エラーのとき終了
    print('data error')
    exit()
#   パルス幅初期値送信
for cnt in range(12):
    readdata = FT_rpi.transfer_dacs(handle, 'Q000'+format(cnt,'x')+("{:03x}".format(pwalist[cnt]))+chr(0xd),9)
#   パルス幅変化時間(移動速度)初期値送信
for cnt in range(12):
    readdata = FT_rpi.transfer_dacs(handle, 'q000'+format(cnt,'x')+'001'+chr(0xd),9)
#   パルス出力開始
readdata = FT_rpi.transfer_dacs(handle, 'Q000F000' + chr(0xd), 9)
time.sleep(0.1)
#   パルス出力変化一時停止
readdata = FT_rpi.transfer_dacs(handle, 'q000S' + chr(0xd), 9)

# コマンドコードのリスト(キー入力チェック用)
comlist = ['W','Q','q','I','y']

# キー入力による操作
while True:
    time.sleep(0.01)

    # キー入力があったときの処理
    if kbwaitf == 2:
        kydatabuf = input()                # enterまでをキー入力
        kydata = kydatabuf.split(',')      # ',' で分離
        kbwaitf = 0                        # キー入力ありを解除

        # enterのみのときは終了
        if len(kydata[0]) == 0: break

        c1 = (kydata[0])[0:1]              # 最初の1文字

        # コマンド文字列の場合
        if c1 in comlist:
            # キー入力データを送信し応答を受信
            tdata = kydata[0] + chr(0xd)
            readdata = FT_rpi.transfer_dacs(handle, tdata, 0)
            print(readdata)                # 応答を画面表示
            continue

        # パルス幅変化時間(移動速度)指定のとき
        if (c1 == 'V' or c1 == 'v') and len(kydata[0]) == 1:
            try:
                s = int(kydata[1], 10)     # 整数に変換
            except:
                print('invalid data')
                continue
            if s > 4095: s = 4095
            if s < 1:    s = 1
            pwavel  = float(s)        # パルス幅変化時間設定
            if pwamov != 0:           # 停止中でないとき、速度補間が無効になるため、
                pwanset = 1           # 次のパルス幅変更ありとする
            continue

        # パルス出力移動開始
        if (c1 == 'G' or c1 == 'g') and len(kydata[0]) == 1:
            pwansta = -1              # 移動状態を次に表示のため無効データとする
            # 次のパルス幅変更なしのとき
            if pwanset == 0:
                readdata = FT_rpi.transfer_dacs(handle, 'Q000F000' + chr(0xd), 9)   # 移動開始
                continue
            # 次のパルス幅変更ありのとき
            readdata = FT_rpi.transfer_dacs(handle, 'q000S' + chr(0xd), 9)    # 一時移動を停止
            if pwamov == 0:           # 停止中のとき
                q = 0.0
                for cnt in range(12): # 合成移動量算出
                    q = q + (float(pwanext[cnt] - pwalist[cnt]))**2
                u = math.sqrt(q)
                re = ''               # 速度補間状態
            else:                     # 移動中または一時停止中
                re = 'xxxxxxxxxxxx'   # 全チャンネルが速度補間無効
            for cnt in range(12):
                if pwamov == 0:       # 停止中のとき
                    t = pwanext[cnt] - pwalist[cnt]    # 移動量算出
                    if t < 0: t = -t
                    if t == 0:
                        p = 1
                        re = '0' + re       # 該当チャンネルは移動なし
                    else:
                        p = int(pwavel * u / float(t))   # パルス幅変化時間(移動速度)算出
                        if p > 4095:
                            p = 4095
                            re = 'x' + re   # 該当チャンネルは速度補間無効
                        else:
                            re = '1' + re   # 該当チャンネルは速度補間有効
                        if p == 0: p = 1
                else:                 # 移動中または一時停止中
                    p = int(pwavel)   # 移動中または一時停止中のときは全チャンネルを同じ速度とする

                # パルス幅変化時間(移動速度)送信
                readdata = FT_rpi.transfer_dacs(handle, 'q000'+format(cnt,'x')+("{:03x}".format(p))+chr(0xd),9)
            print('  ' + re)          # 速度補間状態表示

            for cnt in range(12):     # 目標パルス幅送信 
                readdata = FT_rpi.transfer_dacs(handle, 'Q000'+format(cnt,'x')+("{:03x}".format(pwanext[cnt]))+chr(0xd),9)
                pwalist[cnt] = pwanext[cnt]  # パルス幅現在値更新
            readdata = FT_rpi.transfer_dacs(handle, 'Q000F000' + chr(0xd), 9)   # 移動開始
            print(' start')
            if pwamov == 0: pwanset = 0    # 停止中で移動を開始したときは、
                                           # 速度補間有効にて、次のパルス幅変更なしとする
            continue

        # パルス出力移動一時停止
        if (c1 == 'T' or c1 == 't') and len(kydata[0]) == 1:
            readdata = FT_rpi.transfer_dacs(handle, 'q000S' + chr(0xd), 9)    # 移動一時停止
            print(' force stop')
            continue

        # パルス幅指定またはパルス幅読取りのとき
        try:
            n = int(kydata[0], 10)     # チャンネル番号を整数に変換
        except:
            print('invalid data')
            continue
        if n > 11 or n < 0:
            print('invalid data')
            continue
        pwadisp = n                    # パルス幅表示チャンネルを更新
        # パルス幅読取りのみ
        if len(kydata) == 1:
            continue;
        # パルス幅指定
        try:
            s = int(kydata[1], 10)     # 位置を整数に変換
        except:
            print('invalid data')
            continue
        if s < pwalow[n]:              # 最小値の制限
            s = pwalow[n]
        if s> pwahigh[n]:              # 最大値の制限
            s = pwahigh[n]
        pwanext[n] = s                 # 次のパルス幅を変更
        pwanset = 1                    # 次のパルス幅変更ありとする
        continue;

    # キー入力がないとき
    # パルス出力状態と表示指定チャンネルのパルス幅を読取
    tdata = 'q0R&' + 'Q000'+format(pwadisp,'x')+'R'+ chr(0xd)
    readdata = FT_rpi.transfer_dacs(handle, tdata, 18)   # 送信

    if len(readdata) != 18 or readdata[0:1] != 'n' or readdata[9:10] != 'N':
        # エラーのとき、コマンドを送信し、残留データを読み切る
        time.sleep(1)
        readdata = FT_rpi.transfer_dacs(handle, 'W0000000'+chr(0xd), 0)
        if len(readdata) < 9:          # それでも応答がないとき
            print("")
            print('time-out error')
            break;                     # ループを終了
        else:
            continue;                  # 応答があればループを継続

    # 正常にデータを受信したとき
    # パルス出力状態と表示指定チャンネルのパルス幅を画面表示
    sdata = readdata[2:8]
    if sdata == '000000': pwamov = 0   # 全チャンネル移動終了のとき
    else: pwamov = 1                   # 移動中または一時停止中
    p = int(sdata[3:6],16)
    if p != pwansta:                   # 移動状態の変化があったとき
        print("\r  {:012b}".format(p),'            ')    # 全チャンネルの状態を表示
        pwansta = p                    # 移動状態を更新
    try:
        s = int(readdata[14:17], 16)   # パルス幅を整数に変換
    except:
        continue
    wdisp = '\r  ch'+format(pwadisp,'>2')+' '+format(s,'>4')+' -> '+format(pwanext[pwadisp],'>4')+'  | '
    print(wdisp,end='')                # 表示指定チャンネルのパルス幅(現在->目標)を表示

# デバイスをCLOSE
FT_rpi.close_dacs(handle)
