#******************************************************
#
# 15BXW(15BXP) AD変換 サンプルプログラム
#              計測データ波形表示とファイル保存用
#
#                     ダックス技研株式会社 2024/09/08
#******************************************************

# Windows10(64bit) 、Python 3 にて動作
# DACS USBデバイスドライバをインストール済とします。
# d15adcg.py と FTadc.py を同じディレクトリに格納
# 4ch動作となっていますが、2chの15BXPでも動作します。
# グラフィック matplotlib のインストールが必要です。
#
# デバイスをUSBポートに接続
# d15adcg.py を起動すると
#
# アンプゲインを初期化します。
#    初期値は全チャンネル1倍。デバイスにもセット。
#    gainlist の初期設定値変更にて任意設定可
# 計測間隔の初期設定は変数のみ。 初期値 1ms
#    計測間隔は計測開始時にデバイスにセットします。
#    adcinter の初期設定値変更にて任意設定可
#    (参考)計測間隔はデバイス内部のクロックにより精密制御
#
# d15adcg.py では、計測データ受信処理にかかる時間を短縮し、
# 他の処理に時間を渡すサンプル例として、計測間隔に応じて、
# 複数の計測データ(100ms分)を、一括受信処理しています。
#   計測間隔 100ms以上  1データ
#       例   10ms      10データ一括受信
#            1ms       100データ一括受信
#            0.1ms     1000データ一括受信
#
# 操作方法
# G(enter) とキー入力すると計測を開始し、
#    波形を表示します。
#    計測データを、計測間隔分を１行として、最大900000行
#          ファイル mesdata.txt に保存します。
#          １行は画面表示と同じ4ch分のテキスト形式。
#    4ch分の電圧値を連続して画面に表示します。
#          画面表示は一括受信データの最後のデータ分のみ。
#
# 計測中に、なにかキーを押すと計測を停止します。
# A,x,x,x,x(enter) とキー入力すると
#    ch1、ch2、ch3、ch4 の順でアンプゲインを設定します。
#    x: 1,10,100 のいずれか。数値を省略したチャンネルは変更なし
#    A(enter) とキー入力すると表示のみ
# T,x---x(enter) とキー入力すると計測間隔(変数のみ)を設定します。
#    x--x: 100～10000000 単位マイクロ秒
#    T(enter) とキー入力すると表示のみ
#
# -(マイナス) enter にてプログラム終了

import FTadc as FT
import msvcrt
import time
from matplotlib import pyplot as plt

# 関数 計測開始
#     引数  : 計測間隔(単位マイクロ秒)

def adcstart(ainter):
    # 計測間隔指定 Jコマンド送信(受信あり)
    td = 'J0' + format((ainter//2-1), '06x') + chr(0xd)
    rd = FT.transfer_dacs(handle, td, 0)
    # 計測開始 Sコマンド送信(受信なし)
    td = 'S04F' + chr(0xd)
    rd = FT.transfer_dacs(handle, td, -1)
    return

# 関数 計測停止

def adcstop():
    # 計測停止目的でIコマンドの送信
    td = 'I00186A0' + chr(0xd)
    rd = FT.transfer_dacs(handle, td, -1)
    # 残留データを読切る
    for cnt in range(100):
        rd = FT.transfer_dacs(handle, '', 0)
        if len(rd) == 0:
            break
    return

# 関数 アンプゲイン送信
#     引数   : アンプゲインリスト
#     戻り値 : 受信文字列

def adcamp(glist):
    # 送信用文字列準備
    td = 'G001'
    for cnt in range(4):   # アンプゲインコードに変換
        if glist[3-cnt] == 100:
            td += '2'
        elif glist[3-cnt] == 10:
            td += '1'
        else:
            td += '0'
    td += chr(0xd)
    # Gコマンド送信(受信あり)
    rd = FT.transfer_dacs(handle, td, 0)
    return rd

# 関数 アンプゲインの表示
#     引数1  : アンプゲインリスト
#     引数2  : チャンネル数

def dispgain(glist, num):
    sdata = ''
    for cnt in range(num):
        if glist[cnt] == 100:
            sdata += 'x100'
        elif glist[cnt] == 10:
            sdata += ' x10'
        else:
            sdata += '  x1'
        if cnt != 3:
            sdata += ','
    print('gain = ' + sdata)
    return

# **** ここから開始 ****

# **** 初期設定 ****

# デバイスをOPEN
handle = FT.init_dacs(0)
if handle.value == None:
    print('no device')
    time.sleep(3)
    exit()

# 表示チャンネル数設定(チャンネル数 1～4)
chnum = 4

# 計測停止と受信残留データの読切り
adcstop()                     # 計測停止
adcstartf = 0                 # 停止中とする
print('ADC stop')             # 停止中を表示

# アンプゲイン初期値設定
gainlist = [1, 1, 1, 1]       # アンプゲイン初期値
                              # 1: 1倍  10: 10倍  100: 100倍

# アンプゲインをデバイスに設定
readdata = adcamp(gainlist)
if len(readdata) != 9:        # 受信データ数を確認
    print('device error')     # 15BXW, 15BXP ではないとき
    FT.close_dacs(handle)     # デバイスをCLOSE
    time.sleep(3)
    exit()                    # 計測ユニットでないときは終了
dispgain(gainlist, chnum)     # アンプゲイン表示

# 計測間隔初期値設定および表示
adcinter = 1000               # 1ms
print('interval = ' + str(adcinter) + 'us')
comn = 100000//adcinter       # 計測間隔4ch分を1データとして
                              #   100ms間の一括受信データ数
comnum = comn*18              # 一括受信文字数(1データの文字数18)

# 計測した電圧値の保存領域を準備
adclist = [0.0, 0.0, 0.0, 0.0]
adcdata = [' ',' ',' ',' ']

# 波形画面表示の準備
xsize = 500
X = [0]
Y = [[0.0]*xsize, [0.0]*xsize, [0.0]*xsize, [0.0]*xsize]
Ycolor = ["#2020E0","#E02020","#20E020","#D0D020"]
          # CH1 青    CH2 赤    CH3 緑    CH4 黄
for cnt in range(1,xsize):
    X.append(cnt)
plt.ylim(-10, 10)             # Y表示範囲
plt.xlim(0, xsize)            # X表示範囲
plt.plot(X, Y[0], color="#808080")
plt.grid(color="#D8D8D8")     # グリッド表示
plt.title('15BXW(15BXP) ADC')
plt.pause(1.0)

# 空き時間計測用
margcount  = 0                # カウント用

# **** 計測した電圧値の画面表示を繰返す ****

while True:

    # **** 計測中のとき ****

    if adcstartf == 1:
        # 4ch一括データを受信する
        #     R0xxxxxx(CR)U0xxxxxx(CR) の18文字 * 一括データ数分を受信
        readdatam,recnum,remnum = FT.transferm_dacs(handle, comnum)

        # 受信データがないとき
        if recnum != comnum:
            margcount += 1             # 空き時間計測値を更新
            if margcount > 2000 or recnum != 0:  # デバイス異常
                adcstop()              # 計測停止
                adcstartf = 0          # 停止中とする
                print('')
                print('device error. ADC stop')  # 停止中を表示
                f.close()              # 計測データファイルをclose
                plt.title('ADC device error')
                plt.pause(1.0)
                continue

        # 受信データがあったとき
        else:
            # 受信データのエラーチェック
            p = 0
            for mcnt in range(comn):
                if readdatam[p] != 0x52 or readdatam[p+9] != 0x55:  # R or U
                    adcstop()              # 計測停止
                    adcstartf = 0          # 停止中とする
                    print('')
                    print('ADC-data error. ADC stop')  # 停止中を表示
                    f.close()              # 計測データファイルをclose
                    plt.title('ADC error')
                    plt.pause(1.0)
                    break
                p += 18
            if adcstartf != 1:
                continue

            # 正常にデータを受信したとき
            # 直前の波形表示データを一括受信データ分、前へスクロール
            if comn < xsize:
                xlt = comn
                for cnt in range(xsize):
                    for cntc in range(chnum):
                        Y[cntc][cnt] = Y[cntc][xlt]
                    xlt += 1
                    if xlt >= xsize:
                        break

            pdata = ''                     # 計測データ文字表示用
            for mcnt in range(comn):       # 一括受信データの処理
                p = mcnt*18 + 2                # AD変換値の文字列中の位置
                for cnt in range(2):
                    # 3桁特殊文字列を10進数の整数に変換
                    for cnta in range(2):
                        a = 0
                        for cntb in range(3):  # 1ch分のAD変換値を算出
                            a = a*0x40 + (int(readdatam[p]) -0x30)
                            p += 1
                        # 電圧値保存 (0--0x3FFFF) --> (-10V～+10V)ゲイン1倍換算
                        #     チャンネル順とデータ順の関係  R0(ch3)(ch1) U0(ch4)(ch2)
                        #     adclist にはチャンネル順で電圧値が並ぶ
                        adclist[cnt+(cnta ^ 1)*2] = (float(a - 0x20000))/13107.2
                    p += 3

                if dcount == 0:                # 開始から最初のデータは無効とする
                    for cnt in range(4):
                        adclist[cnt] = 0.0

                # 計測データ(csv形式)をファイル書込み
                if dcount != 0 and dcount <= 900000 or mcnt == comn-1:
                    csvdata = "{:6d}".format(dcount)   # ファイル保存用文字列
                    for cnt in range(chnum):       # 計測データを文字列に
                        if gainlist[cnt] == 1:     # 1倍のとき -10V～+10V
                            adcdata = "{:8.4f}".format(adclist[cnt])
                        elif gainlist[cnt] == 10:  # 10倍のとき -1000mV～+1000mV
                            adcdata = "{:8.2f}".format(adclist[cnt]*100)
                        else:                      # 100倍のとき -100mV～+100mV
                            adcdata = "{:8.3f}".format(adclist[cnt]*10)
                        csvdata += ',' + adcdata
                        if mcnt == comn-1:         # 計測データ文字表示用
                            pdata += '    ' + adcdata
                    if dcount <= 900000:       # 計測データ(csv形式)をファイル書込み
                        f.write(csvdata + '\n')
                if dcount == 900000:
                    f.close()                  # 計測データファイルをclose
                dcount += 1                    # データカウンタ更新
                if dcount > 100000000:
                    dcount = 100000000

                # 波形表示用にデータを格納
                xs = xsize - comn + mcnt
                if xs >= 0:
                    for cnt in range(chnum):
                        Y[cnt][xs] = adclist[cnt]

            # 波形表示(Xサイズ分)
            if remnum < 4000:      # 受信残留データが多いときは波形表示スキップ
                plt.cla()
                for cntc in range(chnum):
                    plt.plot(X, Y[cntc], color=Ycolor[cntc])
                plt.grid(color="#D8D8D8")      # グリッド表示
                plt.ylim(-10, 10)              # Y表示範囲
                plt.xlim(0, xsize)             # X表示範囲
                plt.title('ADC start' + tidata)    # タイトル(計測間隔、アンプゲイン)表示
                plt.pause(0.001)               # 波形画面表示

            # 計測データを文字表示
            pdata += "{:11d}".format(dcount-1)
            print(pdata, end='\r')

            margcount = 0          # 空き時間カウンタをリセット

        time.sleep(0.01)           # 処理時間を空ける

    # **** 停止中のとき ****

    else:
        plt.pause(0.1)                         # 波形画面表示

        # 残留データがあれば読切る
        readdata = FT.transfer_dacs(handle, '', 0) # 受信文字数の指定なしで受信のみ
        time.sleep(0.01)

    # **** キー入力の確認 ****

    if not msvcrt.kbhit():         # キー入力なし
        continue

    # **** キー入力があったときの処理 ****

    # 計測を停止する
    if adcstartf == 1:             # 計測中のとき
        adcstop()                  # 計測停止
        plt.title('ADC stop' + tidata)   # タイトル(計測間隔、アンプゲイン)表示
        plt.pause(0.1)
        adcstartf = 0              # 停止中とする
        print('')
        print('ADC stop')          # 停止中を表示
        f.close()                  # 計測データファイルをclose

    # キー入力文字を取得
    kydatabuf = input()
    # マイナスキー入力のときは終了
    if kydatabuf == '-':
        break
    if len(kydatabuf) != 0:
        kydata = kydatabuf.split(',')  # ',' で分離
        kyNG = 0                       # 仮にキー入力データを正常とする

        # キー入力をコマンド文字列に変換する
        if len(kydata[0]) == 1:        # 最初の文字列は1文字だけ
            c1 = (kydata[0])[0]        # 最初の1文字

            # 計測開始のとき
            if (c1 == 'G' or c1 == 'g') and len(kydata) == 1:
                print('ADC start')     # 計測中を表示
                f = open('mesdata.txt', 'wt')   # 計測データファイルをopen

                # チャンネル番号と電圧単位を画面表示
                pdata = '|'
                for cnt in range(chnum):
                    pdata += 'ch' + str(cnt+1)
                    if gainlist[cnt] == 1:     # 1倍のとき
                        pdata += '    10V |'
                    elif gainlist[cnt] == 10:  # 10倍のとき
                        pdata += ' 1000mV |'
                    else:                      # 100倍のとき
                        pdata += '  100mV |'
                print(pdata + 'data count|')
                tidata = ' ' + str(adcinter) + 'us'   # 波形表示画面タイトル
                tidata += ' Bx' + str(gainlist[0])
                tidata += ' Rx' + str(gainlist[1])
                tidata += ' Gx' + str(gainlist[2])
                tidata += ' Yx' + str(gainlist[3])
                for cntc in range(chnum):
                    for cnt in range(xsize):
                        Y[cntc][cnt] = 0.0     # 波形表示データに0値
                margcount = 0          # 空き時間カウンタをリセット
                adcstartf = 1          # 計測中とする
                dcount = 0             # データカウンタをリセット
                adcstart(adcinter)     # 計測開始

            # アンプゲイン設定のとき
            elif c1 == 'A' or c1 == 'a':
                cntlast = len(kydata)-1
                if cntlast <= 4:       # 4chデータまで
                    # アンプゲイン保存
                    for cnt in range(cntlast):   # アンプゲイン(int)に変換
                        aky = kydata[cnt+1]
                        if aky == '1' or aky == '10' or aky == '100':
                            gainlist[cnt] = int(aky,10)
                        elif aky != '':
                            kyNG = 1   # 不正データ
                            break
                else:
                    kyNG = 1           # 不正データ
                if kyNG == 0:          # 正常データのとき
                    adcamp(gainlist)   # Gコマンド送信
                    dispgain(gainlist, chnum)  # アンプゲインの表示

            # 計測間隔設定のとき
            elif c1 == 'T' or c1 == 't':
                if len(kydata) > 2:
                    kyNG = 1           # 不正データ
                elif len(kydata) == 2:
                    ainter = 0
                    try:
                        ainter = int(kydata[1],10)   # intに変換
                    except:
                        pass
                    if ainter >= 100 and ainter <= 20000000:
                        adcinter = ainter
                        comn = 100000//adcinter      # 一括受信データ数
                        if comn > 1000:
                            comn = 1000
                        if comn == 0:
                            comn = 1
                        comnum = comn*18             # 一括受信文字数
                    else:
                        kyNG = 1       # 不正データ
                if kyNG == 0:          # 正常データのとき計測間隔の表示
                    print('interval = ' + str(adcinter) + 'us')
            else:
                kyNG = 2           # 該当するキーコードなし
        else:
            kyNG = 3               # 先頭が1文字ではない
    else:
        kyNG = 3                   # enterのみ
    if kyNG != 0:                  # キー入力データ不正を表示
        print('invalid data ' + str(kyNG))

# **** デバイスをCLOSE ****

FT.close_dacs(handle)
