# 15BXW(15BXP) AD変換 サンプルプログラム

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# d15adc.py と FT.py を同じディレクトリに格納
# 4ch動作となっていますが、2chの15BXPでも動作します。
#
# デバイスをUSBポートに接続
# d15adc.py を起動すると
#
# アンプゲインを初期化します。
#    初期値は全チャンネル1倍。デバイスにもセット。
#    gainlist の初期設定値変更にて任意設定可
# 計測間隔の初期設定は変数のみ。 初期値 100ms
#    計測間隔は計測開始時にデバイスにセットします。
#    adcinter の初期設定値変更にて任意設定可
#    (参考)計測間隔はデバイス内部のクロックにより精密制御
# 4ch分の電圧値を連続して画面に表示します
#
# 操作方法

# 計測中に、キーボードのenterキーのみを押すと
#    計測を停止してキー入力に移行します。
# A,x,x,x,x(enter) とキー入力すると
#    ch1、ch2、ch3、ch4 の順でアンプゲインを設定します。
#    x: 1,10,100 のいずれか。数値を省略したチャンネルは変更なし
#    A(enter) とキー入力すると表示のみ
# T,x---x(enter) とキー入力すると計測間隔(変数のみ)を設定します。
#    x--x: 100～10000000 単位マイクロ秒
#    T(enter) とキー入力すると表示のみ
#
# W0000000(enter)などのコマンド文字列入力にて
#    デバイスを操作することができます。
#
# enterのみ入力にて電圧値の連続表示を再開
# マイナスキーとenterを入力にてプログラム終了

import FT_rpi
import time

#******** キーボード入力検知処理 ********
import threading
import evdev      # キーボードからのキー入力検出に使用しています。
                  # インストールが必要  例 pip3 install evdev
                  # DACS販売型式 RPi-4B2G には evdev インストール済
kbdevice = evdev.InputDevice('/dev/input/event0')
                  # キーボードからのキー入力検出に使用しています。
                  # キーボードからの入力に移行しない場合は、
                  # event0 を、キーボードに割当てられた番号に変更してください。
                  # 例 event0 event1 event2 event3 など
kbwaitf = 0       # キーボードから入力中  0:入力中ではない, 2:enterキーを押した

# キーボードのenterキーを押したときを検出
def kbhit_read():
    global kbwaitf                              # キーボード入力中をグローバル変数とする
    kbwaitf = 0                                 # 仮に入力中でないとする
    for event in kbdevice.read_loop():
        if event.type == evdev.ecodes.EV_KEY:
            if event.value == 1:                # キーを押したとき
                if event.code == 28:
                    kbwaitf = 2                 # enterキーを押したとき

# キーボードからのキー入力検出 threadを開始
th = threading.Thread(target=kbhit_read, daemon=True)
th.start()

#******** ここからが 計測処理の本題です ********

# 関数 計測開始
#     引数  : 計測間隔(単位マイクロ秒)

def adcstart(ainters):
    # 計測間隔指定 Jコマンド送信(受信あり)
    td = 'J0' + format((ainters//2-1), '06x') + chr(0xd)
    FT_rpi.transfer_dacs(handle, td, 0)
    # 計測開始 Sコマンド送信(受信なし)
    td = 'S04F' + chr(0xd)
    FT_rpi.transfer_dacs(handle, td, -1)
    return

# 関数 計測停止

def adcstop():
    # 計測停止目的でIコマンドの送信(受信なし)
    td = 'I00186A0' + chr(0xd)
    FT_rpi.transfer_dacs(handle, td, -1)
    # 残留データを読切る
    for cnts in range(1000):
        rd = FT_rpi.transfer_dacs(handle, '', 0)
        if cnts > 5:
            if len(rd) == 0:
                break
    return

# 関数 アンプゲイン送信
#     引数   : アンプゲインリスト
#     戻り値 : 受信文字列

def adcamp(glists):
    # 送信用文字列準備
    td = 'G001'
    for cnts in range(4):   # アンプゲインコードに変換
        if glists[3-cnts] == 100:
            td = td + '2'
        elif glists[3-cnts] == 10:
            td = td + '1'
        else:
            td = td + '0'
    td = td + chr(0xd)
    # Gコマンド送信(受信あり)
    rd = FT_rpi.transfer_dacs(handle, td, 0)
    return rd

# 関数 アンプゲインの表示
#     引数1  : アンプゲインリスト
#     引数2  : チャンネル数

def dispgain(glists, num):
    sdatas = ""
    for cnts in range(num):
        if glists[cnts] == 100:
            sdatas = sdatas + 'x100'
        elif glists[cnts] == 10:
            sdatas = sdatas + ' x10'
        else:
            sdatas = sdatas + '  x1'
        if cnts != 3:
            sdatas = sdatas + ','
    print('gain = ' + sdatas)
    return

# 関数 チャンネル番号と電圧単位を画面表示
#     引数1  : アンプゲインリスト
#     引数2  : チャンネル数

def dispchnum(glists, num):
    pdatas = '|'
    for cnts in range(num):
        if glists[cnts] == 1:     # 1倍のとき
            pdatas = pdatas + 'ch' + format(cnts+1,'x') +'    10V |'
        elif glists[cnts] == 10:  # 10倍のとき
            pdatas = pdatas + 'ch' + format(cnts+1,'x') +' 1000mV |'
        else:                     # 100倍のとき
            pdatas = pdatas + 'ch' + format(cnts+1,'x') +'  100mV |'
    print(pdatas + 'data count|')
    return

# **** ここから開始 ****

# **** 初期設定 ****

# デバイスをOPEN
handle = FT_rpi.init_dacs(0)
if handle.value == None:
    print('no device')
    exit()

# 表示チャンネル数設定(チャンネル数 1～4)
chnum = 4

# 計測停止と受信残留データの読切り
#     スタート前に計測を開始ししていたときの対策
adcstop()                     # 計測を停止し、受信残留データの読切り

# アンプゲイン初期値設定
gainlist = [1, 1, 1, 1]       # アンプゲイン初期値
                              # 1: 1倍  10: 10倍  100: 100倍
# アンプゲインをデバイスに設定
readdata = adcamp(gainlist)
if len(readdata) != 9:        # 受信データ数を確認
    print('device error')     # 15BXW, 15BXP ではないとき
    FT_rpi.close_dacs(handle) # デバイスをCLOSE
    exit()                    # 計測ユニットでないときは終了
dispgain(gainlist, chnum)     # アンプゲイン表示

# 計測間隔初期値設定および表示
adcinter = 100000             # 100ms
print('interval = ' + ("{:d}".format(adcinter)) + 'us')

# 計測した電圧値の保存領域を準備
adclist = [0.0, 0.0, 0.0, 0.0]

# コマンドコードのリスト(キー入力チェック用)
comlist = ['W','S','I','J','G']

# 計測開始
adcstartf = 1                 # 計測中とする
dcount = 0                    # データカウンタをリセット
print('ADC start')            # 計測中を表示
dispchnum(gainlist, chnum)    # チャンネル番号と電圧単位を画面表示
adcstart(adcinter)            # 計測開始

# **** 計測した電圧値の画面表示を繰り返す ****

while True:

    # **** 計測中のとき ****
    if adcstartf == 1:
        # 4ch分のデータを受信する(受信文字数18指定の受信のみ)
        #     R0xxxxxx(CR)U0xxxxxx(CR) の18文字を受信
        readdata = FT_rpi.transfer_dacs(handle, '', 18)

        # 受信データがないとき
        if len(readdata) == 0:
            time.sleep(0.01)

        # 受信データがあったとき
        else:
            # 受信データがエラーのとき
            if len(readdata) != 18 or readdata[0:1] != 'R' or readdata[9:10] != 'U':
                adcstartf = 0          # 停止中とする
                print('')
                print('Rx data error, ADC stop')   # 受信データエラーによる停止を表示
                adcstop()              # 計測停止および残留データの処理
                print('push enter key')
                continue

            # 正常にデータを受信したとき
            sdata = ''                 # 画面表示用の文字列
            for cnt in range(2):
                p = cnt*9 + 2   # AD変換値の文字列中位置
                # 3桁特殊文字列を10進数の整数に変換
                for cnta in range(2):
                    a = 0
                    for cntb in range(3):
                        a = a*0x40 + (ord(readdata[(p):(p+1)]) -0x30)
                        p += 1
                    # 電圧値保存 (0--0x3FFFF) --> (-10V～+10V)ゲイン1倍換算
                    #     チャンネル順とデータ順の関係  R0(ch3)(ch1) U0(ch4)(ch2)
                    #     adclist にはチャンネル順で電圧値が並ぶ
                    adclist[cnt+(cnta ^ 1)*2] = 10.0*float(a - 0x20000)/(float(0x20000))

            # 電圧値を画面表示
            dcount += 1                # データカウンタ更新
            if dcount > 100000000:
                dcount = 100000000
            pdata = ''
            for cnt in range(chnum):
                if gainlist[cnt] == 1:     # 1倍のとき -10V～+10V
                    pdata = pdata + "{:11.4f}".format(adclist[cnt]) + ' '
                elif gainlist[cnt] == 10:  # 10倍のとき -1000mV～+1000mV
                    pdata = pdata + "{:11.2f}".format(adclist[cnt]*100) + ' '
                else:                      # 100倍のとき -100mV～+100mV
                    pdata = pdata + "{:11.3f}".format(adclist[cnt]*10) + ' '
            if kbwaitf != 2:
                print(pdata + "{:10d}".format(dcount), end='\r')

    # **** 停止中のとき ****
    else:
        # 残留データがあれば読切る
        readdata = FT_rpi.transfer_dacs(handle, '', 0)  # 受信文字数の指定なしで受信のみ

    if kbwaitf != 2:     # キーボードからの入力中でないとき
        continue
    # enterキーを押してないときは、ここまでの画面表示を繰り返す

    # キーボードのenterキーを押したとき
    kydatabuf = input()    # 最初のenterキーまでを無効データとしてを読み捨てる

    # 以下のキー入力処理を繰り返す

    while True:

        # 計測中のときは計測を停止する
        if adcstartf == 1:             # 計測中のとき
            adcstartf = 0              # 停止中とする
            print('')
            print('ADC stop')          # 停止中を表示
            adcstop()                  # 計測停止

        kydatabuf = input('\r' + 'keyin = ')    # キー入力
        clen = len(kydatabuf)
        c0 = " "

        # キー入力中にenterキーのみ入力のとき
        if clen == 0:
            dcount = 0                 # データカウンタをリセット
            adcstartf = 1              # 計測中とする
            print('ADC start')         # 計測中を表示
            dispchnum(gainlist, chnum) # チャンネル番号と電圧単位を画面表示
            kbwaitf = 0                # キー入力中を解除
            adcstart(adcinter)         # 計測開始
            break                      # 計測結果の連続表示へ

        # キー入力最初の1文字
        c0 = kydatabuf[0]
        if c0 == '-':        # マイナスキーのときはプログラム終了へ
            break

        # キー入力文字を取得
        kydata = kydatabuf.split(',')  # ',' で分離
        c1 = (kydata[0])[0:1]          # 分離データの最初の1文字
        kyNG = 0                       # 仮にキー入力データを正常とする

        # コマンド文字列の場合
        if c1 in comlist and len(kydata[0]) > 1:
            # キー入力データを送信し応答を受信
            tdata = kydata[0] + chr(0xd)
            readdata = FT_rpi.transfer_dacs(handle, tdata, 0)
            if len(readdata) > 0:      # 応答を画面表示
                print('\r' + kydatabuf + ' --> ' + readdata)
            else:
                print('time-out')      # 応答がないとき

        # キー入力をコマンド文字列に変換する
        elif len(kydata[0]) == 1:      # 最初の文字列は1文字だけ

            # アンプゲイン設定のとき
            if c1 == 'A' or c1 == 'a':
                cntlast = len(kydata)-1
                if cntlast <= 4:       # 4chデータまで
                    # アンプゲイン保存
                    for cnt in range(cntlast):   # アンプゲイン(int)に変換
                        aky = kydata[cnt+1]
                        if aky == '1' or aky == '10' or aky == '100':
                            gainlist[cnt] = int(aky,10)
                        elif aky != '':
                            kyNG = 1   # 不正データ
                            break
                else:
                    kyNG = 1           # 不正データ
                if kyNG == 0:          # 正常データのとき
                    adcamp(gainlist)   # Gコマンド送信
                    dispgain(gainlist, chnum)  # アンプゲインの表示

            # 計測間隔設定のとき
            elif c1 == 'T' or c1 == 't':
                if len(kydata) > 2:
                    kyNG = 1           # 不正データ
                elif len(kydata) == 2:
                    ainter = 0
                    try:
                        ainter = int(kydata[1],10)   # intに変換
                    except:
                        pass
                    if ainter >= 100 and ainter <= 20000000:
                        adcinter = ainter
                    else:
                        kyNG = 1       # 不正データ
                if kyNG == 0:          # 正常データのとき計測間隔の表示
                    print('interval = ' + ("{:d}".format(adcinter)) + 'us')

            else:
                kyNG = 2               # 該当するキーコードなし
        else:
            kyNG = 3                   # 先頭が1文字ではない

        if kyNG != 0:                  # キー入力データ不正を表示
            print('invalid data ' + ("{:d}".format(kyNG)))
            
    if c0 == '-':    # 最初の文字がマイナスキーはプログラム終了
        break

# **** デバイスをCLOSE ****

FT_rpi.close_dacs(handle)
