# DACS-2500KB-SCN3 カウンタ サンプルプログラム

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# d15cnt_rpi.py と FT_rpi.py を同じディレクトリに格納

# デバイスをUSBポートに接続
# d15cnt_rpi.py を起動すると

# カウント開始コマンドを自動送信します。
# カウンタ値読取りコマンドを自動送信し、
#     カウンタ値を読取って連続表示します。
# 0～2番のカウンタ値とホールド値を表示します。

# キーボードのenterキーのみを押すとキー入力に移行します。

# 1 enter とキー入力すると
#    0～5番のカウンタ値表示に変わります。
# 0 enter とキー入力すると
#    0～2番のカウンタ値とホールド値に戻ります。
# W0000000(enter)などのコマンド文字列入力にて
#    キー入力したコマンド文字列を送信し
#        受信した応答データを表示します。
#    デバイスを操作することができます。
#    例: M018(enter) と入力すると
#    カウンタ0番がエンコーダモードになります。

# enterのみ入力にてカウンタ値表示を再開
# マイナスキーとenterを入力にてプログラム終了

import FT_rpi
import time

#******** キーボード入力検知処理 ********
import threading
import evdev      # キーボードからのキー入力検出に使用しています。
                  # インストールが必要  例 pip3 install evdev
                  # DACS販売型式 RPi-4B2G には evdev インストール済
kbdevice = evdev.InputDevice('/dev/input/event0')
                  # キーボードからのキー入力検出に使用しています。
                  # キーボードからの入力に移行しない場合は、
                  # event0 を、キーボードに割当てられた番号に変更してください。
                  # 例 event0 event1 event2 event3 など
kbwaitf = 0       # キーボードから入力中  0:入力中ではない, 2:enterキーを押した

# キーボードのenterキーを押したときを検出
def kbhit_read():
    global kbwaitf                              # キーボード入力中をグローバル変数とする
    kbwaitf = 0                                 # 仮に入力中でないとする
    for event in kbdevice.read_loop():
        if event.type == evdev.ecodes.EV_KEY:
            if event.value == 1:                # キーを押したとき
                if event.code == 28:
                    kbwaitf = 2                 # enterキーを押したとき

# キーボードからのキー入力検出 threadを開始
th = threading.Thread(target=kbhit_read, daemon=True)
th.start()

#******** ここからが カウンタの本題です ********

# デバイスをOPEN
handle = FT_rpi.init_dacs(0)
if handle.value == None:
    print('no device')
    exit()
    
# カウント開始を送信
writedata = 'M008&M028&M048&m008&m028&m048' + chr(0xd)
readdata = FT_rpi.transfer_dacs(handle, writedata, 54)
if len(readdata) != 54:     # (9文字)X(6コマンド)=54
    print('time-out error') # タイムアウトエラー
    FT_rpi.close_dacs(handle)   # 接続終了
    exit()                  # 応答がないときは終了

# 3ch(0～2番)カウンタ値と3ch(0～2番)ホールド値を取得するコマンド文字列を準備
c1command = 'M00&M01&M02&M03&M04&M05&M06&M07&M08&M09&M0A&M0B&W0R' + chr(0xd)
# 3ch(0～5番)カウンタ値を取得するコマンド文字列を準備
c2command = 'M00&M01&M02&M03&M04&M05&m00&m01&m02&m03&m04&m05&W0R' + chr(0xd)

dispmode = 0                # 表示内容選択  0:disp1  1:disp2
dcommand = c1command        # 送信要求コマンド文字列をdisp1用とする

disp1 = '|    ch0    |    ch1    |    ch2    |    hd0    |    hd1    |    hd2    |'
disp2 = '|    ch0    |    ch1    |    ch2    |    ch3    |    ch4    |    ch5    |'
print(disp1)                # カウンタ番号を画面表示

c1 = ' '                    # 応答1番目の文字

# コマンドコードのリスト(キー入力チェック用)
comlist = ['W','P','M','m','T','Y','I']

# カウンタ値の取得と画面表示を繰返す
while True:
    time.sleep(0.01)

    # カウンタ値とデジタル入力を取得。受信文字数：(9文字)X(13コマンド)=117
    readdata = FT_rpi.transfer_dacs(handle, dcommand, 117)

    if len(readdata) != 117 or readdata[0:1] != 'N':
        # エラーのとき、コマンドを送信し、残留データを読み切る
        time.sleep(1)
        readdata = FT_rpi.transfer_dacs(handle, 'W0R'+chr(0xd), 0)
        if len(readdata) < 9: # それでも応答がないとき
            print("")
            print('time-out error')
            break;          # ループを終了
        else:
            continue;       # 応答があればループを継続

    # 正常にデータを受信したとき
    sdata = ''              # 画面表示用の文字列を初期化

    for cnt in range(6):
        p = cnt*18 + 4      # カウンタ値文字列中の位置

        # 16進数下位4桁文字列を10進数の整数に変換
        try:
            cdata = int(readdata[(p):(p+4)],16)
        except ValueError:
            break

        # 16進数上位4桁文字列を10進数の整数に変換
        p += 9
        try:
            hdata = int(readdata[(p):(p+4)],16)
        except ValueError:
            break
        cdata = cdata + hdata*0x10000

        # 32bit長の負数処理
        if cdata > 0x7fffffff:
            cdata = cdata - 0x100000000

        # 画面表示用に12桁の文字列で連結する
        sdata = sdata + ("{:12d}".format(cdata))

    # カウンタ値を画面表示

    if kbwaitf != 2:     # キーボードからの入力中でないとき
        if len(sdata) == 72:    # 正常データのとき
            print('\r' + sdata + '  ',end='\r')
        else:                   # 不正データのとき
            print('data error'+' '*62)
            time.sleep(2)
        continue
    # enterキーを押してないときは、ここまでの画面表示を繰り返す

    # キーボードのenterキーを押したとき
    kydatabuf = input()    # 最初のenterキーまでを無効データとしてを読み捨てる

    # 以下のキー入力処理を繰り返す

    while True:
        kydatabuf = input('\r' + 'keyin = ')

        # キー入力中にenterキーのみ入力のとき
        if len(kydatabuf) == 0:
            kbwaitf = 0        # キー入力中を解除
            # カウンタ番号を再表示して連続表示に戻る
            if dispmode == 0:
                print('\r' + disp1)
            else:
                print('\r' + disp2)
            break

        # 最初の1文字
        c1 = kydatabuf[0]
        if c1 == '-':          # マイナスキーのときはプログラム終了へ
            break

        # コマンド文字列の場合
        if c1 in comlist and len(kydatabuf) > 1:
            # キー入力データを送信し応答を受信
            tdata = kydatabuf+chr(0xd)
            readdata = FT_rpi.transfer_dacs(handle, tdata, 0)
            print('\r' + kydatabuf + ' --> ' + readdata)     # 応答を画面表示
            continue

        # 先頭が0または1
        if c1 == '0':
            dispmode = 0
            dcommand = c1command       # 送信要求文字列をdisp1用に
        elif c1 == '1':
            dispmode = 1
            dcommand = c2command       # 送信要求文字列をdisp2用に
        # 入力データが不正のとき
        else:
            print('invalid data')

    if c1 == '-':    # 最初の文字がマイナスキーはプログラム終了
        break

# デバイスをCLOSE
FT_rpi.close_dacs(handle)
