# DACS-2500KB PWMパルス出力 サンプルプログラム

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# Drsw_rpi_sample.py、Drsw_rpi.py、FT_rpi.py を同じディレクトリに格納

# デバイスをUSBポートに接続
# Drsw_rpi_sample.py を起動すると
# PWMパルス幅読取りコマンドを自動送信し、
#   12チャンネル分のパルス幅(単位:クロック数)を読取って、
#   12チャンネルのパルス幅変化状態と共に連続表示します。

# マウスのセンターキー(スクロールキー)を押すとキー入力に移行します。

# 以下のキー入力後、enterにて指示データを送信
# /(スラッシュ)で複数の指示データを連結可能  例 W,0,100/W,1,200/G

#     G       パルス出力開始
#     N       パルス出力終了
#     I,y--y,z--z  周期設定
#             y--y  クロック周波数設定(MHz) 下記のいずれか
#                    0.5, 1, 2, 4, 8, 16, 32, 64
#             z--z  周期設定 小数点可 (単位: μs)
#     IP,y--y,z--z  周期設定
#             y--y  クロック周波数設定(MHz) 下記のいずれか
#                    0.5, 1, 2, 4, 8, 16, 32, 64
#             z--z  周期設定 (単位: クロック数)
#     W,xx,y--y  目標パルス幅設定
#             xx   チャンネル番号 (0～11)  全チャンネル指定(12)
#             y--y パルス幅 小数点可 (単位: μs)
#     WP,xx,y--y  目標パルス幅設定
#             xx   チャンネル番号 (0～11)  全チャンネル指定(12)
#             y--y パルス幅 (単位: クロック数)
#     V,xx,y--y  変化速度設定
#             xx   チャンネル番号 (0～11)  全チャンネル指定(12)
#             y--y 変化速度 小数点可 (単位: パルス幅 μs / s)
#     VP,xx,y--y  変化速度設定
#             xx   チャンネル番号 (0～11)  全チャンネル指定(12)
#             y--y 変化速度 (単位: クロック数 / s)
#     S       パルス幅変化停止
#               停止後のパルス幅を目標パルス幅とし、停止を解除
#     DP,y--y デジタル出力極性(16進数6桁 bit23～0)
#     DO,y--y デジタル出力(16進数6桁 bit23～0)

# 0～11 を入力してパルス幅 μs 表示チャンネルを指定します
# enterのみ入力にてパルス幅表示を再開
# -(マイナス) enter にてプログラム終了

# L/z--z/xx,y--y/xx,y--y/  --  /x,y--y   同時スタート&ストップ
#     z--z  変化速度 (単位: μs / s) 小数点可
#     xx    チャンネル番号 0～11
#     y--y  目標パルス幅 (単位: μs) 小数点可
#     対象となるチャンネルのデータを /（スラッシュ）で連結
# LP/z--z/xx,y--y/xx,y--y/  --  /x,y--y  同時スタート&ストップ
#     z--z  変化速度 (単位: クロック数 / s)
#     y--y  目標パルス幅 (単位: クロック数)

import Drsw_rpi
import time
import shutil     # ターミナルサイズ取得用

terminal_size = shutil.get_terminal_size()
if terminal_size.columns < 90:      # ターミナル横サイズが不足
    print('check terminal_size ---->')
    time.sleep(3)
    exit()

#******** マウスのセンターキー(スクロールキー)検知処理 ********
import threading
import evdev      # マウスのセンターキー検出に使用しています。
                  # インストールが必要  例 pip3 install evdev
                  # DACS販売型式 RPi-4B2G には evdev インストール済
kbdevice = evdev.InputDevice('/dev/input/event0')
                  # キーボード入力に移行しない場合は、
                  # event0 を、マウスに割当てられた番号に変更してください。
                  # 例 event0 event1 event2 event3 など
kbwaitf = 0       # キーボードから入力中  0:入力中ではない, 1:入力中

# マウスのセンターキーを押したときを検出
def kbhit_read():
    global kbwaitf                          # キーボード入力中をグローバル変数とする
    kbwaitf = 0                             # 仮に入力中でないとする
    for event in kbdevice.read_loop():
        if event.type == evdev.ecodes.EV_KEY:
            if event.value == 1:            # キーを押したとき
                if event.code == evdev.ecodes.BTN_MIDDLE:   # センターキーを押した
                    kbwaitf = 1             # キーボード入力中とする

# キーボードから指示データ入力
def kbinput_read():
    t0 = time.time()
    ret = ''
    while True:
        kybuf = input('keyin = ')
        if (time.time() - t0) > 0.3:        # 新しく入力したデータ
            ret = kybuf
            break
        else:                               # 入力開始以前のデータを破棄
            print('\033[1A')                # 表示ラインを戻す
    return ret

# マウスのセンターキー検出 threadを開始
th = threading.Thread(target=kbhit_read, daemon=True)
th.start()

#******** ここからが PWMパルス出力の本題です ********

# デバイスをOPEN、PWMパルス出力ボードを確認
ret = Drsw_rpi.Drsw_init(9)
if ret[0] == 'OK':
    handle = ret[1]     # デバイスハンドルをセット
    print('OK')
else:
    print('no device')
    time.sleep(3)
    exit()

interclk = 99.0                     # クロック周波数設定(MHz)(未設定値)
internum = 20000                    # 周期(クロック数)
inters   = 20000.0                  # 周期(μs)
dispch   = 0                        # パルス幅表示チャンネル
print('clock= ***MHz   interval= ***.**us')
disp1 = '|ch0 |  1 |  2 |  3 |  4 |  5 |  6 |  7 |  8 |  9 | 10 | 11 | s |  DI  | '
print(disp1 + ' ch0')               # チャンネル番号0～11を画面表示

# 12チャンネル分のパルス幅 画面表示を繰返す
while True:
    time.sleep(0.01)

    # パルス幅を取得
    readdata = Drsw_rpi.Drsw_read(handle, '12')
    # パルス幅変化状態とデジタル入力を取得
    sreaddata = Drsw_rpi.Drsw_sread(handle)

    # 12チャンネル分のパルス幅とステータスを表示
    if len(readdata) == 59 and len(sreaddata) == 10:  # 正常データのとき
        pdata = readdata.split(',')  # ',' で分離
        sdata = ''
        for cnt in range(12):        # 1行に連結
            sdata = sdata + ' ' + pdata[cnt]
        qdata = sreaddata.split(',') # ',' で分離
        sdata = sdata + ' ' + qdata[0] + ' ' + qdata[1]
        if interclk > 90:            # クロック周波数/周期未設定のとき
            sdata = sdata + ' ***.**us'
        else:                        # 表示指定チャンネルのパルス幅(μs)
            p = float(int(pdata[dispch],16))/interclk
            if p > 100.0:
                ii = format(p,'.2f')
            else:
                ii = format(p,'.3f')
            sdata = sdata + ' ' + ii
            if p > inters:
                sdata = sdata + ' over'  # パルス幅が周期を超えている
            elif int(pdata[dispch],16) == 0xffff:
                sdata = sdata + ' limit' # パルス幅上限
            else:
                sdata = sdata + 'us'     # 正常なパルス幅範囲
        if kbwaitf == 0:             # キーボードからの入力がないとき
            print(sdata + '     ',end='\r')
    else:                            # 不正データのとき
        print('data error' + ' '*70)
        time.sleep(2)

    if kbwaitf == 0:     # キーボードからの入力中でないとき
        continue

# キーボードから指示データを入力
    print()
    kydatabuf = kbinput_read()
    kbwaitf = 0
    # マイナスキー入力のときは終了
    if kydatabuf == '-':
        break
    if len(kydatabuf) != 0:
        # パルス幅表示チャンネルの指定
        c1 = kydatabuf[0:1]
        if c1 >= '0' and c1 <= '9':
            try:
                dch = int(kydatabuf)
            except ValueError:
                dch = 99
            if dch >= 0 and dch < 12:
                dispch = dch
            else:
                print('\033[31m'+kydatabuf+'-->'+'NG'+'\033[0m')
        else:
            # 同時スタート&ストップ
            if c1.upper() == 'L':
                readdata = Drsw_rpi.Drsw_pmove(handle, kydatabuf)
            # 指示データ送信
            else:
                readdata = Drsw_rpi.Drsw_write(handle, kydatabuf)
            # 応答結果を表示
            if readdata == 'OK':
                print(kydatabuf+'-->'+readdata)
            else:
                print('\033[31m'+kydatabuf+'-->'+readdata+'\033[0m')

    # クロック周波数、周期を表示
    ret = Drsw_rpi.Drsw_iread()
    interclk = ret[0]               # クロック周波数設定(MHz)
    internum = ret[1]               # 周期(クロック数)
    if interclk > 90.0:             # クロック周波数/周期未設定のとき
        print('clock= ***MHz   interval= ***.**us')
    else:                           # クロック周波数/周期設定済のとき
        inters = float(internum)/interclk    # 周期(μs)
        print('clock = ' + format(interclk,'.1f') + 'MHz',end='')
        if inters > 100.0:
            ii = format(inters,'.2f')
        else:
            ii = format(inters,'.3f')
        print('   interval = '+ ii + 'us  ' + format(internum,'05x'))
    # チャンネル番号を表示
    print(disp1 + ' ch' + format(dispch,'d')) 
    continue

# デバイスをCLOSE
Drsw_rpi.Drsw_close(handle)
