#******************************************************
#
# 15BXW(15BXP) AD変換 サンプルプログラム
#
#                     ダックス技研株式会社 2021/01/24
#******************************************************

# Windows10(64bit) 、Python 3 にて動作
# DACS USBデバイスドライバをインストール済とします。
# d15adc.py と FT.py を同じディレクトリに格納
# 4ch動作となっていますが、2chの15BXPでも動作します。
#
# デバイスをUSBポートに接続
# d15adc.py を起動すると
#
# アンプゲインを初期化します。
#    初期値は全チャンネル1倍。デバイスにもセット。
#    gainlist の初期設定値変更にて任意設定可
# 計測間隔の初期設定は変数のみ。 初期値 100ms
#    計測間隔は計測開始時にデバイスにセットします。
#    adcinter の初期設定値変更にて任意設定可
#    (参考)計測間隔はデバイス内部のクロックにより精密制御
#
# 操作方法
# G(enter) とキー入力すると計測を開始し、
#    4ch分の電圧値を連続して画面に表示します。
# 計測中に、なにかキーを押すと計測を停止します。
# A,x,x,x,x(enter) とキー入力すると
#    ch1、ch2、ch3、ch4 の順でアンプゲインを設定します。
#    x: 1,10,100 のいずれか。数値を省略したチャンネルは変更なし
#    A(enter) とキー入力すると表示のみ
# T,x---x(enter) とキー入力すると計測間隔(変数のみ)を設定します。
#    x--x: 100～10000000 単位マイクロ秒
#    T(enter) とキー入力すると表示のみ
#
# W0000000(enter)などのコマンド文字列入力にて
#    デバイスを操作することができます。
#
# enter のみをキー入力するとプログラム終了

import FT
import msvcrt
import time

# 関数 計測開始
#     引数  : 計測間隔(単位マイクロ秒)

def adcstart(ainter):
    # 計測間隔指定 Jコマンド送信(受信あり)
    td = 'J0' + format((ainter//2-1), '06x') + chr(0xd)
    rd = FT.transfer_dacs(handle, td, 0)
    # 計測開始 Sコマンド送信(受信なし)
    td = 'S04F' + chr(0xd)
    rd = FT.transfer_dacs(handle, td, -1)
    return

# 関数 計測停止

def adcstop():
    # 計測停止目的でIコマンドの送信
    td = 'I00186A0' + chr(0xd)
    rd = FT.transfer_dacs(handle, td, -1)
    # 残留データを読切る
    for cnt in range(100):
        rd = FT.transfer_dacs(handle, '', 0)
        if len(rd) == 0:
            break
    return

# 関数 アンプゲイン送信
#     引数   : アンプゲインリスト
#     戻り値 : 受信文字列

def adcamp(glist):
    # 送信用文字列準備
    td = 'G001'
    for cnt in range(4):   # アンプゲインコードに変換
        if glist[3-cnt] == 100:
            td = td + '2'
        elif glist[3-cnt] == 10:
            td = td + '1'
        else:
            td = td + '0'
    td = td + chr(0xd)
    # Gコマンド送信(受信あり)
    rd = FT.transfer_dacs(handle, td, 0)
    return rd

# 関数 アンプゲインの表示
#     引数1  : アンプゲインリスト
#     引数2  : チャンネル数

def dispgain(glist, num):
    sdata = ""
    for cnt in range(num):
        if glist[cnt] == 100:
            sdata = sdata + 'x100'
        elif glist[cnt] == 10:
            sdata = sdata + ' x10'
        else:
            sdata = sdata + '  x1'
        if cnt != 3:
            sdata = sdata + ','
    print('gain = ' + sdata)
    return

# **** ここから開始 ****

# **** 初期設定 ****

# デバイスをOPEN
handle = FT.init_dacs(0)
if handle.value == None:
    print('no device')
    exit()

# 表示チャンネル数設定(チャンネル数 1～4)
chnum = 4

# 計測停止と受信残留データの読切り
adcstop()                     # 計測停止
adcstartf = 0                 # 停止中とする
print('ADC stop')             # 停止中を表示

# アンプゲイン初期値設定
gainlist = [1, 1, 1, 1]       # アンプゲイン初期値
                              # 1: 1倍  10: 10倍  100: 100倍

# アンプゲインをデバイスに設定
readdata = adcamp(gainlist)
if len(readdata) != 9:        # 受信データ数を確認
    print('device error')     # 15BXW, 15BXW ではないとき
    FT.close_dacs(handle)     # デバイスをCLOSE
    exit()                    # 計測ユニットでないときは終了
dispgain(gainlist, chnum)     # アンプゲイン表示

# 計測間隔初期値設定および表示
adcinter = 100000             # 100ms
print('interval = ' + ("{:d}".format(adcinter)) + 'us')

# 計測した電圧値の保存領域を準備
adclist = [0.0, 0.0, 0.0, 0.0]

# コマンドコードのリスト(キー入力チェック用)
comlist = ['W','S','I','J','G']

# **** 計測した電圧値の画面表示を繰返す ****

while True:

    # **** 計測中のとき ****

    if adcstartf == 1:
        # 4ch分のデータを受信する(受信文字数18指定の受信のみ)
        #     R0xxxxxx(CR)U0xxxxxx(CR) の18文字を受信
        readdata = FT.transfer_dacs(handle, '', 18)

        # 受信データがないとき
        if len(readdata) == 0:
           time.sleep(0.01)

        # 受信データがあったとき
        else:
            # 受信データがエラーのとき
            if len(readdata) != 18 or readdata[0:1] != 'R' or readdata[9:10] != 'U':
                adcstop()              # 計測停止
                adcstartf = 0          # 停止中とする
                print('')
                print('ADC stop')      # 停止中を表示
                continue

            # 正常にデータを受信したとき
            sdata = ''                 # 画面表示用の文字列
            for cnt in range(2):
                p = cnt*9 + 2   # AD変換値の文字列中位置
                # 3桁特殊文字列を10進数の整数に変換
                for cnta in range(2):
                    a = 0
                    for cntb in range(3):
                        a = a*0x40 + (ord(readdata[(p):(p+1)]) -0x30)
                        p += 1
                    # 電圧値保存 (0--0x3FFFF) --> (-10V～+10V)ゲイン1倍換算
                    #     チャンネル順とデータ順の関係  R0(ch3)(ch1) U0(ch4)(ch2)
                    #     adclist にはチャンネル順で電圧値が並ぶ
                    adclist[cnt+(cnta ^ 1)*2] = 10.0*float(a - 0x20000)/(float(0x20000))

            # 電圧値を画面表示
            dcount += 1                # データカウンタ更新
            if dcount > 100000000:
                dcount = 100000000
            pdata = ''
            for cnt in range(chnum):
                if gainlist[cnt] == 1:     # 1倍のとき -10V～+10V
                    pdata = pdata + "{:11.4f}".format(adclist[cnt]) + ' '
                elif gainlist[cnt] == 10:  # 10倍のとき -1000mV～+1000mV
                    pdata = pdata + "{:11.2f}".format(adclist[cnt]*100) + ' '
                else:                      # 100倍のとき -100mV～+100mV
                    pdata = pdata + "{:11.3f}".format(adclist[cnt]*10) + ' '
            print(pdata + "{:10d}".format(dcount), end='\r')

    # **** 停止中のとき ****

    else:
        # 残留データがあれば読切る
        readdata = FT.transfer_dacs(handle, '', 0)  # 受信文字数の指定なしで受信のみ

    # **** キー入力があったときの処理 ****

    if msvcrt.kbhit():

        # 計測を停止する
        if adcstartf == 1:             # 計測中のとき
            adcstop()                  # 計測停止
            print('')
            print('ADC stop')          # 停止中を表示
        adcstartf = 0                  # 停止中とする

        # キー入力文字を取得
        kydatabuf = input()
        kydata = kydatabuf.split(',')  # ',' で分離
        c1 = (kydata[0])[0:1]          # 最初の1文字
        kyNG = 0                       # 仮にキー入力データを正常とする

        # コマンド文字列の場合
        if c1 in comlist and len(kydata[0]) > 1:
            # キー入力データを送信し応答を受信
            tdata = kydata[0] + chr(0xd)
            readdata = FT.transfer_dacs(handle, tdata, 0)
            if len(readdata) > 0:
                print(readdata)        # 応答を画面表示
            else:
                print('time-out')      # 応答がないとき

        # enterのみのときは終了
        elif len(kydata[0]) == 0:
            break

        # キー入力をコマンド文字列に変換する
        elif len(kydata[0]) == 1:      # 最初の文字列は1文字だけ

            # 計測開始のとき
            if (c1 == 'G' or c1 == 'g') and len(kydata) == 1:
                adcstart(adcinter)     # 計測開始
                adcstartf = 1          # 計測中とする
                print('ADC start')     # 計測中を表示

                # チャンネル番号と電圧単位を画面表示
                pdata = '|'
                for cnt in range(chnum):
                    if gainlist[cnt] == 1:     # 1倍のとき
                        pdata = pdata + 'ch' + format(cnt+1,'x') +'    10V |'
                    elif gainlist[cnt] == 10:  # 10倍のとき
                        pdata = pdata + 'ch' + format(cnt+1,'x') +' 1000mV |'
                    else:                      # 100倍のとき
                        pdata = pdata + 'ch' + format(cnt+1,'x') +'  100mV |'
                print(pdata + 'data count|')
                dcount = 0             # データカウンタをリセット

            # アンプゲイン設定のとき
            elif c1 == 'A' or c1 == 'a':
                cntlast = len(kydata)-1
                if cntlast <= 4:       # 4chデータまで
                    # アンプゲイン保存
                    for cnt in range(cntlast):   # アンプゲイン(int)に変換
                        aky = kydata[cnt+1]
                        if aky == '1' or aky == '10' or aky == '100':
                            gainlist[cnt] = int(aky,10)
                        elif aky != '':
                            kyNG = 1   # 不正データ
                            break
                else:
                    kyNG = 1           # 不正データ
                if kyNG == 0:          # 正常データのとき
                    adcamp(gainlist)   # Gコマンド送信
                    dispgain(gainlist, chnum)  # アンプゲインの表示

            # 計測間隔設定のとき
            elif c1 == 'T' or c1 == 't':
                if len(kydata) > 2:
                    kyNG = 1           # 不正データ
                elif len(kydata) == 2:
                    ainter = 0
                    try:
                        ainter = int(kydata[1],10)   # intに変換
                    except:
                        pass
                    if ainter >= 100 and ainter <= 20000000:
                        adcinter = ainter
                    else:
                        kyNG = 1       # 不正データ
                if kyNG == 0:          # 正常データのとき計測間隔の表示
                    print('interval = ' + ("{:d}".format(adcinter)) + 'us')

            else:
                kyNG = 2               # 該当するキーコードなし
        else:
            kyNG = 3                   # 先頭が1文字ではない

        if kyNG != 0:                  # キー入力データ不正を表示
            print('invalid data ' + ("{:d}".format(kyNG)))

# **** デバイスをCLOSE ****

FT.close_dacs(handle)
