# DACS-2500K(KB) モーションコントローラ

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# Dpmv_rpi.py と FT_rpi.py を同じディレクトリに格納

import FT_rpi
import time

axmvlist = [0,0,0,0,0,0]   # マスター軸指定用各軸移動量
ax6mode = 0                # 第6軸独立動作(仮になしとする)

def Dpmv_init(devnum):
# 関数 デバイスの初期化
#    引数   : デバイス番号(int型)
#    戻り値 : (1)'OK' 正常終了 'NG' 異常 ,(2)デバイスハンドル
#    各軸移動量初期値をセット
#      電源投入時に、デバイス側でも初期化をしますが、
#      デバイスを再接続しないでプログラムを再起動した場合の対策です。
#    第6軸独立動作の設定があれば継続します。

    global ax6mode

    ret = 'NG'            # 仮に異常終了とする
    for cnt in range(4):  # 4デバイス分の検索
        if devnum >= 9:
            devn = cnt
        else:             # デバイス番号を指定している
            devn = devnum
        handle = FT_rpi.init_dacs(devn)
        if handle.value != None:
            # デジタル入力を実行
            readdata = FT_rpi.transfer_dacs(handle, 'W0Rxxxxx'+chr(0xd), 9)
            if (len(readdata) == 9) and (readdata[0:1] == 'R'):
                # qコマンドの応答を確認('q0R'のRはPWMデバイスのq設定を避けるため)
                readdata = FT_rpi.transfer_dacs(handle, 'q0R'+chr(0xd), 9)
                if len(readdata) == 9 and readdata[0:1] == 's':
                    readdata = ''
                    for cntc in range(6):
                        rdata = FT_rpi.transfer_dacs(handle, 'P0' + format(cntc,'x') +'00000'+chr(0xd), 9)
                        if len(rdata) == 9:
                            readdata = readdata + rdata
                        else:
                            break
                    if len(readdata) == 54:     # 6コマンド分の受信データ数を確認
                        # 第6軸独立動作設定の確認
                        readdata = FT_rpi.transfer_dacs(handle, 'Q06'+chr(0xd), 9)
                        if len(readdata) == 9:
                            ret = 'OK'
                            if readdata[4:5] != '0':
                                ax6mode = 1     # 第6軸独立動作とする
                readdata = FT_rpi.transfer_dacs(handle, 'W0Rxxxxx'+chr(0xd), 9)
                if ret == 'OK':
                    break
            FT_rpi.close_dacs(handle)   # デバイスをCLOSE
        if devnum < 9:
            break
    return ret,handle

def Dpmv_read(handle, rmode):
# 関数 移動量または位置の取得
#    引数   : デバイスハンドル, 読取モード(M:移動量 P:位置)
#    戻り値 : 文字列
#               移動量(または位置) 1軸から6軸(10進数)をカンマ区切り,
#               デジタル入力(24bit 16進6桁) 
    sdata = ''
    if rmode == 'M' or rmode == 'P':
        if rmode == 'M':
            # 移動量読取りコマンド文字列を準備
            dcommand = 'Q00&Q01&Q02&Q03&Q04&Q05&W0R' + chr(0xd)
        else:
            # 位置読取りコマンド文字列を準備
            dcommand = 'q00&q01&q02&q03&q04&q05&W0R' + chr(0xd)

        # 移動量または位置を取得。受信文字数：(9文字)X(7コマンド)=63
        readdata = FT_rpi.transfer_dacs(handle, dcommand, 63)

        if len(readdata) != 63 or (readdata[0:1] != 'S' and readdata[0:1] != 's'):
            # エラーのとき、コマンドを送信し、残留データを読み切る
            time.sleep(1)
            readdata = FT_rpi.transfer_dacs(handle, 'W0000000'+chr(0xd), 0)
        else:
            # 正常にデータを受信したとき
            for cnt in range(7):
                if cnt < 6:         # 移動量または位置のとき
                    # 16進数5桁文字列を10進数の整数に変換
                    p = cnt*9 + 3   # 移動量または位置の文字列中位置
                    try:
                        cdata = int(readdata[(p):(p+5)],16)
                    except ValueError:
                        break
                    # 負方向の処理
                    if (cdata & 0x80000) != 0:
                        if rmode == 'M':
                            # 移動量の場合: bit19が符号で以下は絶対値
                            cdata = 0x80000 - cdata
                        else:
                            # 位置の場合  : 20bitの純2進数
                            cdata = cdata - 0x100000
                    # カンマで連結する
                    sdata = sdata + ("{:d}".format(cdata)) + ','
                else:               # デジタル入力のとき
                    sdata = sdata + readdata[56:62]
            if cnt != 6:
                sdata = ''          # データ異常のとき
    return sdata

def Dpmv_sread(handle):
# 関数 ステータスの取得
#    引数   : デバイスハンドル
#    戻り値 : 文字列 x,y,z--z
#                x:1～6軸(第6軸独立のときは1～5)の状態
#                    G:移動中  D:ドウェル時間中  /:待機  S:停止
#                y:第6軸独立動作の状態
#                    N:第6軸独立なし G:移動中  D:ドウェル時間中  /:待機  S:停止
#                z--z: ステータス (16進下位5桁)
    sdata = ''
    readdata = FT_rpi.transfer_dacs(handle, 'Q06'+chr(0xd), 9)
    if len(readdata) == 9 and readdata[0:1] == 'S' and readdata[2:3] == '6':
        try:
            a = int(readdata[4:8],16)    # intに変換
            if a & 0x1c != 0:            # 強制、リミット、異常停止
                sdata = 'S,'
            elif a & 3 == 3:             # 移動中
                sdata = 'G,'
            elif a & 3 == 1:             # ドウェル時間中
                sdata = 'D,'
            else:
                sdata = '/,'             # 待機
            if a & 0x1000 != 0:          # 第6軸独立
                if a & 0x800:            #   強制、リミット停止
                    sdata = sdata + 'S,'
                elif a & 0x300 == 0x300: #   移動中
                    sdata = sdata + 'G,'
                elif a & 0x100 == 0x100: #   ドウェル時間中
                    sdata = sdata + 'D,'
                else:
                    sdata = sdata + '/,' #   待機
            else:
                sdata = sdata + 'N,'     # 第6軸独立なし
            sdata = sdata + readdata[3:8]
        except:
            sdata = ''              # 受信なしのとき
    return sdata

def Dpmv_write(handle, wdata):
# 関数 コマンド送信とレスポンス受信
#    引数   : デバイスハンドル, 指示データ
#    戻り値 : 'OK' 正常終了 'NG' 異常終了
#       G GC GD 移動開始の場合 'OKx'  x:マスター軸番号1～6
#       複数指示データ連結では、最後が移動開始指示のとき軸番号付加
#       データ連結では、一部のデータが不正のとき、すべての転送なし

# 指示データ(文字列)
# /(スラッシュ)で複数の指示データを連結可能  例 1,100/2,-1000
#     x,y--y 各軸の移動量を指定
#          x:軸番号(1～6)  y--y:指定移動量(整数 単位 パルス)
#          x=9 のとき、全軸に同じ移動量をセット
#     V,y--y 移動速度を指定(小数点可 単位 Hz)
#     A,y--y       y--y:加速度を指定(小数点可 単位 KHz/s)
#                  台形加減速
#     A,y--y,z--z  y--y:加速度を指定(小数点可 単位 KHz/s)
#                  z--z:S字時間を指定(小数点可 単位 s)
#     G 移動開始   GC エンドレス移動開始
#     DT,y--y ドウエルタイム設定  y--y:ドウェルタイム(ms)
#     GD ドウェルタイム後の移動開始
#     S 強制停止
#     R 位置リセット
#     DH,xxx デジタル出力(bit23～12) 16進数3桁で指定
#     LL,xxxxx リミットスイッチ low ON  16進数5桁で指定
#     LH,xxxxx リミットスイッチ High ON 16進数5桁で指定
#                DACS-2500KB-PM63 取扱説明書 リミット入力信号欄を参照
#     DP,xxx パルス出力と移動方向出力極性 16進数3桁で指定
#                DACS-2500KB-PM63 取扱説明書 出力極性指定欄を参照
#     WY ウオッチドクタイマー有効   WN 無効
#     6Y 第6軸独立動作指定  6N 第6軸独立動作解除
#     6V,y--y 第6軸独立動作の速度を指定
#     6A,y--y 第6軸独立動作の加速度を指定 台形加減速
#     6A,y--y,z--z 第6軸独立動作の加速度を指定 S字加減速
#     6G 第6軸独立移動開始  6GC 第6軸独立エンドレス移動開始
#     6DT,y--y 第6軸独立動作ドウエルタイム設定
#     6GD 第6軸独立動作 ドウェルタイム後の移動開始
#     6S 第6軸独立移動強制停止

    global axmvlist                     # マスター軸指定用各軸移動量
    global ax6mode                      # 第6軸独立動作

    ret = 'NG'                          # 仮に異常終了とする
    cdatalist = wdata.split('/')        # '/' で分離
    alltdata = ''                       # コマンド文字列の準備

    for cntlist in range(len(cdatalist)):  # 指示データ数分の繰返し
        cdata = cdatalist[cntlist].split(',')     # ',' で分離
        clen = len(cdata)               # ','区切りのデータ数
        c0len = len(cdata[0])
        if c0len != 0:
            c1 = (cdata[0])[0:1].upper()    # 最初の1文字
        else:
            alltdata = ''
            break

        # コマンド文字列に変換

        tdata = ''                  # 1指示データ分のコマンド文字列の準備

        # 先頭が1～6の軸番号指定。9は全軸指定
        if clen == 2 and ((c1 >= '1' and c1 < '7') and (c0len == 1) or c1 == '9'):
            try:
                a = int(cdata[1],10)    # intに変換
                b = abs(a)              # 絶対値
                if b > 0x7ffff:         # 最大値の制限
                    b = 0x7ffff
                c = b
                if a < 0:               # 負数のとき
                    c = c + 0x80000     # bit20 に負の符号
                sdata = format(c, '05x')  # 16進数5桁に変換

                if c1 == '9':           # 全軸の移動量指定コマンドを準備
                    tdata = 'P0'
                    for cnt in range(6):
                        tdata = tdata + format(cnt, 'X') + sdata
                        axmvlist[cnt] = b  # 指定移動量(絶対値)を保存
                        if cnt == 5:
                            tdata = tdata + '&'
                        else:
                            tdata = tdata + '&P0'
                else:                   # 指定軸の移動量指定コマンドを準備
                    axis = int(c1,10) - 1   # 軸番号(整数0～5)に変換
                    tdata = 'P0' + format(axis, 'X') + sdata + '&'
                    axmvlist[axis] = b  # 指定移動量(絶対値)を保存
            except:
                tdata = ''              # 不正データのとき

        # 速度指定
        elif c1 == 'V':
            tdata = Dpmv_velset(cdata)
            if len(tdata) != 0:         # 速度指定コマンドを準備
                tdata = 'P08' + tdata + '&'

        # 加速度指定
        elif c1 == 'A':
            tdata = Dpmv_accset(cdata)
            if len(tdata) != 0:         # 加速度指定コマンドを準備
                tdata = 'P09' + tdata + '&'

        # 軸移動開始
        elif c1 == 'G':
            # 指定移動量が最大の軸を判定
            a = 0 
            axis = 0
            cntlast = 6
            if ax6mode == 1:
                cntlast = 5             # 第6軸独立設定中
            for cnt in range(cntlast):
                if axmvlist[cnt] > a:
                    axis = cnt
                    a = axmvlist[cnt]

            # 軸移動開始コマンドを準備
            if c0len == 2:
                c2 = (cdata[0])[1:2].upper()
                if c2 == 'C':           # エンドレス移動
                    tdata = 'Q0F' + format(axis,'x') + '&'
                elif c2 == 'D':         # ドウェルタイム後の移動
                    tdata = 'Q08' + format(axis + 8,'x') + '&'
            else:                       # 通常移動
                tdata = 'Q08' + format(axis,'x') + '&'

        # 軸移動強制停止
        elif c1 == 'S':
            # 軸移動停止コマンドを準備
            tdata = 'Q09&'

        # 位置リセット
        elif c1 == 'R':
            # パルス分配異常リセット、位置リセットコマンドを準備
            tdata = 'Q0A&Q0B&'

        # デジタル出力(bit23～12)設定
        elif clen == 2 and c1 == 'D' and c0len == 2:
            c2 = (cdata[0])[1:2].upper()
            if c2 == 'H':
                if len(cdata[1]) == 3:
                    tdata = 'W0'+ cdata[1] + 'xxx&'
        # パルス出力極性指定
            elif c2 == 'P':
                if len(cdata[1]) == 3:
                    tdata = 'P0C00'+ cdata[1] + '&'
        # ドウエルタイム
            elif c2 == 'T':
                try:
                    a = int(cdata[1],10)    # intに変換 単位ms
                    if a > 16383:
                        a = 16383 
                    if a >= 0:
                        tdata = 'P0A' + format(a,'05x') + '&'
                except:
                    tdata = ''              # 不正データのとき

        # リミットスイッチ設定
        elif clen == 2 and  c1 == 'L' and c0len == 2:
            if len(cdata[1]) == 5:
                c2 = (cdata[0])[1:2].upper()
                if c2 == 'L':
                    tdata = 'Q0D' + cdata[1] + '&'
                elif c2 == 'H':
                    tdata = 'Q0E' + cdata[1] + '&'

        # ウオッチドグタイマ
        elif c1 == 'W' and c0len == 2:
            c2 = (cdata[0])[1:2].upper()
            if c2 == 'N':      # ウオッチドグタイマ解除
                tdata = 'P0B0&'
            elif c2 == 'Y':    # ウオッチドグタイマ設定
                tdata = 'P0B1&'

        # 第6軸の独立動作
        elif c1 == '6' and c0len >= 2:
            c2 = (cdata[0])[1:2].upper()
            if c2 == 'N':      # 第6軸独立解除
                tdata = 'Z0F0&'
                ax6mode = 0
            elif c2 == 'Y':    # 第6軸独立設定
                tdata = 'Z0F1&'
                ax6mode = 1
            elif c2 == 'V':    # 第6軸独立動作の速度設定
                tdata = Dpmv_velset(cdata)
                if len(tdata) != 0:
                    tdata = 'Z08' + tdata + '&'
            elif c2 == 'A':    # 第6軸独立動作の加速度設定
                tdata = Dpmv_accset(cdata)
                if len(tdata) != 0:
                    tdata = 'Z09' + tdata + '&'
            elif clen == 2 and c2 == 'D':    # 第6軸独立動作のドウェルタイム設定
                if c0len == 3:
                    c3 = (cdata[0])[2:3].upper()
                    if c3 == 'T':
                        try:
                            a = int(cdata[1],10)  # intに変換 単位ms
                            if a > 16383:
                                a = 16383 
                            if a >= 0:
                                tdata = 'Z0A' + format(a,'05x') + '&'
                        except:
                            tdata = ''            # 不正データのとき
            elif c2 == 'G' and ax6mode == 1:      # 第6軸独立移動開始
                if c0len == 3:
                    c3 = (cdata[0])[2:3].upper()
                    if c3 == 'C':                 # エンドレス移動
                        tdata = 'Z003&'
                    elif c3 == 'D':               # ドウェルタイム後の移動
                        tdata = 'Z002&'
                else:
                    tdata = 'Z001&'
            elif c2 == 'S':                       # 第6軸独立移動停止
                tdata = 'Z000&'

        if len(tdata) != 0:
            alltdata = alltdata + tdata     # コマンド文字列の連結
        else:
            alltdata = ''
            break

    # 正常な入力データのときコマンドを送信
    if len(alltdata) != 0:
        num = alltdata.count('&')       # コマンド数
        alltdata = alltdata[:-1] + chr(0xd)  # 末尾の'&'文字をCRに置換
        readdata = FT_rpi.transfer_dacs(handle, alltdata, 0)
        if len(readdata) == num*9:
            ret = 'OK'                  # 正常終了とする
            if c1 == 'G':               # 軸移動開始のときはマスター軸番号を追加
                ret = ret + format(axis+1,'x')
    return ret

def Dpmv_moveA(handle, adata):
# 関数 全軸MOVE (位置指定 MD, 移動量指定 MI)
#    引数   : デバイスハンドル, 指示データ
#    戻り値 : 'OKx' 正常終了   x:マスター軸番号1～6
#             'NG' データ異常 'ER' 送受信異常 'NA' 移動中につき不可

# 指示データ(文字列)  'uu/z--z/x,y--y/x,y--y/  --  /x,y--y'
#                        uu    移動モード
#                              MD : 位置指定  MI : 移動量指定
#                              ドウェルタイム終了後に移動開始 MDW:位置指定 MIW:移動量指定
#                        z--z  移動速度(小数点可 単位 Hz)
#                        x     軸番号 1～6
#                        y--y  移動量または目標位置 整数 (単位: パルス数)
#                        対象となる軸のデータを /（スラッシュ）で連結

    ret = 'NG'                      # 仮にデータ異常とする
    cdatalist = adata.split('/')    # '/' で分離
    cdatanum = len(cdatalist) - 2   # 対象軸数
    if cdatanum < 1 or cdatanum > 6:
        return ret                  # データ不足または過多

    cmode = cdatalist[0].upper()
    if cmode != 'MD' and cmode != 'MI' and cmode != 'MDW' and cmode != 'MIW':
        return ret                  # MOVEコマンドではない

    sdata = Dpmv_sread(handle)      # ステータス読取り
    if len(sdata) != 9:
        return 'ER'                 # 送受信異常とする
    if sdata[0:1] == 'G':
        return 'NA'                 # 移動中につき不可

    try:
        velof = float(cdatalist[1]) # 指定速度
    except ValueError:
        return ret                  # データ異常

    axnolist = [0,0,0,0,0,0]
    axmplist = [0,0,0,0,0,0]
    for cnt in range(cdatanum):     # 対象チャンネル数分の繰返し
        axdata = cdatalist[cnt+2].split(',')    # ',' で分離
        if len(axdata) != 2:
            return ret              # データ異常
        try:
            axnolist[cnt] = int(axdata[0])-1    # 軸番号
            if axnolist[cnt] < 0:
                return ret                      # データ異常
            if axnolist[cnt] == 5 and sdata[2:3] != 'N':  # 第6軸独立動作中
                return 'NA'                     # 移動中につき不可
            axmplist[cnt] = int(axdata[1])      # 移動量または位置
        except ValueError:
            return ret              # データ異常

    if cmode == 'MD' or cmode == 'MDW':
        readdata = Dpmv_read(handle, 'P')   # 現在位置読取り
        p = readdata.split(',')     # ',' で分離
        if len(p) != 7:
            return 'ER'             # 送受信異常とする
        for cnt in range(cdatanum):     # 移動量に変換
            axmplist[cnt] = axmplist[cnt] - int(p[axnolist[cnt]])

    tdata = 'V,' + format(velof,'.2f')  # 移動速度を指示データに
    for cnt in range(6):            # 移動量を指示データに
        mcheck = 0
        for cntcd in range(cdatanum):     # 該当する軸データあり?
            if axnolist[cntcd] == cnt:
                tdata = tdata + '/' + str(cnt+1) + ',' + str(axmplist[cntcd])
                mcheck = 1
                break
        if mcheck == 0:
            tdata = tdata + '/' + str(cnt+1) + ',' + '0'
    if cmode == 'MIW' or cmode == 'MDW':
        tdata = tdata + '/GD'       # ドウェルタイム終了後に移動開始
    else:
        tdata = tdata + '/G'        # 直ちに移動開始
    readdata = Dpmv_write(handle, tdata)   # データ送信
    if readdata[0:2] == 'OK':
        ret = readdata              # 正常終了とする
    else:
        ret = 'ER'                  # 送受信異常とする
    return ret

def Dpmv_move6(handle, data6):
# 関数 6軸単独MOVE (位置指定 MD, 移動量指定 MI)
#    引数   : デバイスハンドル, 指示データ
#    戻り値 : 'OK' 正常終了 'NG' データ異常 'ER' 送受信異常 'NA' 移動中につき不可

# 指示データ(文字列)  'uuu/z--z/y--y'
#                        uuu   移動モード
#                              MD6 : 位置指定  MI6 : 移動量指定
#                              ドウェルタイム終了後に移動開始 MD6W:位置指定 MI6W:移動量指定
#                        z--z  移動速度(小数点可 単位 Hz)
#                        y--y  移動量または目標位置 整数 (単位: パルス数)

    global ax6mode                  # 第6軸独立動作

    ret = 'NG'                      # 仮にデータ異常とする
    cdatalist = data6.split('/')    # '/' で分離
    if len(cdatalist) != 3:
        return ret                  # データ不足または過多
    cmode = cdatalist[0].upper()
    if cmode != 'MD6' and cmode != 'MI6' and cmode != 'MD6W' and cmode != 'MI6W':
        return ret                  # 6軸単独MOVEコマンドではない

    sdata = Dpmv_sread(handle)      # ステータス読取り
    if len(sdata) != 9:
        return 'ER'                 # 送受信異常とする
    c1 = sdata[0:1]
    c2 = sdata[2:3]
    if c2 == 'G' or (c2 == 'N' and c1 != '/' and c1 != 'S'):
        return 'NA'                 # 移動中につき不可

    try:
        velof = float(cdatalist[1]) # 指定速度
    except ValueError:
        return ret                  # データ異常
    try:
        axmp = int(cdatalist[2])    # 移動量または位置
    except ValueError:
        return ret                  # データ異常
    if cmode == 'MD6' or cmode == 'MD6W':
        readdata = Dpmv_read(handle, 'P')   # 現在位置読取り
        p = readdata.split(',')     # ',' で分離
        if len(p) != 7:
            return 'ER'             # 送受信異常とする
        axmp = axmp - int(p[5])     # 移動量に変換

    if c2 == 'N':
        ax6mode = 1                 # 第6軸独立動作とする
        tdata = '6Y/6V,'
    else:
        tdata = '6V,'
    tdata = tdata + format(velof,'.2f')  # 移動速度を指示データに
    tdata = tdata + '/6,' + str(axmp)    # 移動量を指示データ
    if cmode == 'MI6W' or cmode == 'MD6W':
        tdata = tdata + '/6GD'      # ドウェルタイム終了後に移動開始
    else:
        tdata = tdata + '/6G'       # 直ちに移動開始
    readdata = Dpmv_write(handle, tdata)   # データ送信
    if readdata == 'OK':
        ret = readdata              # 正常終了とする
    else:
        ret = 'ER'                  # 送受信異常とする
    return ret

def Dpmv_close(handle):
# 関数 デバイスをCLOSE
#    引数   : デバイスハンドル
#    戻り値 : なし

    FT_rpi.close_dacs(handle)

def Dpmv_velset(cdata):
# 関数 速度設定
#     引数   : 指示データ
#     戻り値 : 送信文字列

    tdata = ''
    if len(cdata) == 2: 
        try:                        # データの単位は Hz
            v = (float(cdata[1]))   # floatに変換
            a = int(v*4.0+0.001)    # intに変換(単位0.25Hz)
            if a > 0xf4240:         # 最大値の制限
                a = 0xf4240
            if a < 1:               # 最小値の制限
                a = 1
            tdata = format(a, '05x')  # 16進数5桁に変換
        except:
            tdata = ''              # 不正データのとき
    return tdata

def Dpmv_accset(cdata):
# 関数 加速度設定
#     引数   : 指示データ
#     戻り値 : 送信文字列

    tdata = ''
    if len(cdata) >= 2: 
        try:                        # データの単位は KHz/s
            v = (float(cdata[1]))   # floatに変換
            a = int(v*0.8+0.001)    # intに変換(単位1.25KHz/s)
            if a > 0xfff:           # 最大値の制限
                a = 0xfff
            if a < 1:               # 最小値の制限
                a = 1
            sdata = format(a, '04x')  # 16進数4桁に変換

            cnt = 0
            if len(cdata) == 3:     # S字指定のとき
                v = (float(cdata[2]))   # S字時間をfloatに変換(単位 s)
                v = v*1100
                q = 6.4             # S字時間最小単位(ms)
                for cnt in range(11):  # S字時間をS字コードに変換
                   if v < q:
                       break
                   q = q *2
            tdata = format(cnt, 'X') + sdata
        except:
            tdata = ''              # 不正データのとき
    return tdata
