# DACS-2500KB-RSV3 PWMパルス出力 サンプルプログラム

# Raspberry pi の Python にて動作
# DACS USBデバイスドライバをインストール済とします。
# d15rcp_rpi.py と FT_rpi.py を同じディレクトリに格納

# デバイスをUSBポートに接続
# d15rcp_rpi.py を起動すると

# Aグループ、Bグループ共に以下の初期設定をします。
#    パルス出力の周波数
#    全チャンネルのパルス幅
#    パルス出力の開始

# 指定パルス幅とデジタル入力状態を連続して表示します。

# キーボードのenterキーのみを押すとキー入力に移行します。

# キー操作にて任意のチャンネルのパルス幅を変更できます。
# A,x,y--y(enter) Aグループ指定
#            x : 0～11 チャンネル番号
#            y--y : ±パルス数(中心位置からのパルス幅偏移量)
#            ,y--y を省略すると表示チャンネル変更のみ
# B,x,y--y(enter) Bグループ指定

# W0000000(enter)などのコマンド文字列入力にて
#    デバイスを操作することができます。

# enterのみ入力にて表示を再開
# マイナスキーとenterを入力にてプログラム終了

# 上記の操作にて指定したチャンネルの指定パルス幅とデジタル入力状態を
# 連続して表示します。
#    表示例 A xch width= aaaa   DI=FFFFFF   B xch width= bbbb   state= 000000
#    x : 上記操作で指定したチャンネル
#    aaaa : Aグループの指定パルス幅   bbbb : Bグループの指定パルス幅
#    DI=FFFFFF : デジタル入力状態 16進数表示
#    state=000000 : 各チャンネルパルス幅変化状態 16進数表示

# デジタル入力により、Aグループの指定パルス幅を増減操作できます。
#    操作対象チャンネルは上記操作で指定したチャンネルとなります。
#    デジタル入力lowにて増減します。
#    この機能が不要の場合は、変数 pmen = 0 としてください。
#
#    デジタル入力の読取り周期ごとに下記パルス幅を増減します。
#        デジタル入力bit0  +1
#                    bit1  +2
#                      ---
#                    bit7  +128
#        デジタル入力bit8  -1
#                    bit9  -2
#                      ---
#                    bit15 -128

import FT_rpi
import time

#******** キーボード入力検知処理 ********
import threading
import evdev      # キーボードからのキー入力検出に使用しています。
                  # インストールが必要  例 pip3 install evdev
                  # DACS販売型式 RPi-4B2G には evdev インストール済
kbdevice = evdev.InputDevice('/dev/input/event0')
                  # キーボードからのキー入力検出に使用しています。
                  # キーボードからの入力に移行しない場合は、
                  # event0 を、キーボードに割当てられた番号に変更してください。
                  # 例 event0 event1 event2 event3 など
kbwaitf = 0       # キーボードから入力中  0:入力中ではない, 2:enterキーを押した

# キーボードのenterキーを押したときを検出
def kbhit_read():
    global kbwaitf                              # キーボード入力中をグローバル変数とする
    kbwaitf = 0                                 # 仮に入力中でないとする
    for event in kbdevice.read_loop():
        if event.type == evdev.ecodes.EV_KEY:
            if event.value == 1:                # キーを押したとき
                if event.code == 28:
                    kbwaitf = 2                 # enterキーを押したとき

# キーボードからのキー入力検出 threadを開始
th = threading.Thread(target=kbhit_read, daemon=True)
th.start()

#******** ここからが PWMパルス出力制御の本題です ********

# パルス幅制限値、中心位置、初期位置を準備
#   Aグループ ch0 --> ch11 の順
pwalow  = [560,560,560,560,560,560,560,560,560,560,560,560]             # 最小値
pwahigh = [2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480] # 最大値
pwacent = [1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520] # 中心値
pwalist = [1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520] # 初期位置
#   Bグループ ch12 --> ch23 の順
pwblow  = [560,560,560,560,560,560,560,560,560,560,560,560]             # 最小値
pwbhigh = [2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480,2480] # 最大値
pwbcent = [1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520] # 中心値
pwblist = [1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520,1520] # 初期位置

# デバイスをOPEN
handle = FT_rpi.init_dacs(0)
if handle.value == None:
    print('no device')
    exit()

# 周波数を初期設定 (内部カウントクロック 1MHz / パルス周波数 50Hz)
#   Aグループ ch0-->ch11
readdata = FT_rpi.transfer_dacs(handle, 'Q0B04E1F' + chr(0xd), 9)
if len(readdata) != 9:          # 最初の通信でPWMボードの確認
    FT_rpi.close_dacs(handle)   # 受信エラーのとき終了
    print('data error')
    exit()
#   Bグループ ch12-->ch23
readdata = FT_rpi.transfer_dacs(handle, 'Q0B14E1F' + chr(0xd), 9)

#   Aグループ パルス幅送信
for cnt in range(12):
    readdata = FT_rpi.transfer_dacs(handle, 'Q000'+format(cnt,'x')+("{:03x}".format(pwalist[cnt]))+chr(0xd),9)
#   Bグループ パルス幅送信
for cnt in range(12):
    readdata = FT_rpi.transfer_dacs(handle, 'Q001'+format(cnt,'x')+("{:03x}".format(pwblist[cnt]))+chr(0xd),9)

# パルス出力開始
#   Aグループ
readdata = FT_rpi.transfer_dacs(handle, 'Q000F000' + chr(0xd), 9)
#   Bグループ
readdata = FT_rpi.transfer_dacs(handle, 'Q001F000' + chr(0xd), 9)
print('PWM output start')

# デジタル入力によるパルス幅の増減操作
pmch = 0         # Aグループ操作対象チャンネル
accf = 0         # パルス幅増減値
pmen = 1         # 操作有効
pbch = 0         # Bグループ表示チャンネル

c1 = ' '         # 応答1番目の文字

# コマンドコードのリスト(キー入力チェック用)
comlist = ['W','Q','q','I','y']

# パルス幅送信とキー入力による操作
while True:
    time.sleep(0.01)
    
    # デジタル入力読取りとステータス読取りを送信
    readdata = FT_rpi.transfer_dacs(handle, 'W0R&q0R'+chr(0xd), 0)

    if len(readdata) < 9 or readdata[0] != 'R':
        # エラーのとき、コマンドを送信し、残留データを読み切る
        time.sleep(1)
        readdata = FT_rpi.transfer_dacs(handle, 'W0R'+chr(0xd), 0)
        if len(readdata) < 9:          # それでも応答がないとき
            print("")
            print('time-out error')
            break                      # ループを終了
        else:
            continue                   # 応答があればループを継続

    # 正常にデータを受信したとき

    if pmen == 1:                      # 増減操作有効のとき
        distate = int(readdata[6:8],16)    # 受信したデジタル入力(bit3-0)を整数に
        accf = (distate ^ 0xff)            # 論理を反転しパルス幅の増方向に
        distate = int(readdata[4:6],16)    # 受信したデジタル入力(bit7-4)を整数に
        accf = accf - (distate ^ 0xff)     # 論理を反転しパルス幅の減方向に

        pwalist[pmch] = pwalist[pmch] + accf   # パルス幅を更新
        if pwalist[pmch] > pwahigh[pmch]:
            pwalist[pmch] = pwahigh[pmch]      # パルス幅最大値
        if pwalist[pmch] < pwalow[pmch]:
            pwalist[pmch] = pwalow[pmch]       # パルス幅最小値

        # 送信文字列準備
        tdata = 'Q000' + format(pmch,'x') + ("{:03x}".format(pwalist[pmch])) + chr(0xd)
        # パルス幅を送信
        readtmp = FT_rpi.transfer_dacs(handle, tdata, 9)

    # 状態を画面表示(パルス幅増減対象チャンネル、指定パルス幅、デジタル入力 --- の順)

    if kbwaitf != 2:     # キーボードからの入力中でないとき
        # Aグループパルス幅増減対象チャンネル + デジタル入力
        sdata = 'A '+format(pmch,'d')+'ch width='+("{:5d}".format(pwalist[pmch]))+'   DI='+readdata[2:8] 
        # Bグループパルス幅指定チャンネル
        sdata = sdata + '   B '+format(pbch,'d')+'ch width='+("{:5d}".format(pwblist[pbch]))
        # ステータス(パルス幅変化状態)
        if len(readdata) == 18 and readdata[9] == 'n':
            sdata = sdata + '   state= ' + readdata[11:17]
        print('\r' + sdata + '  ',end='\r')
        continue
    # enterキーを押してないときは、ここまでの状態画面表示を繰り返す

    # キーボードのenterキーを押したとき
    kydatabuf = input()    # 最初のenterキーまでを無効データとしてを読み捨てる

    # 以下のキー入力処理を繰り返す

    while True:
        kydatabuf = input('\r' + 'keyin = ')

        # キー入力中にenterキーのみ入力のとき
        if len(kydatabuf) == 0:
            kbwaitf = 0        # キー入力中を解除し連続表示に戻る
            break

        # 最初の1文字
        c1 = kydatabuf[0]
        if c1 == '-':          # マイナスキーのときはプログラム終了へ
            break

        # コマンド文字列の場合
        if c1 in comlist and len(kydatabuf) > 1:
            # キー入力データを送信し応答を受信
            tdata = kydatabuf+chr(0xd)
            readdata = FT_rpi.transfer_dacs(handle, tdata, 0)
            print('\r' + kydatabuf + ' --> ' + readdata)     # 応答を画面表示
            continue

        kydata = kydatabuf.split(',')      # ',' で分離

        # 先頭がAまたはB
        if (c1 == 'A' or c1 == 'a') and len(kydata[0]) == 1:
            tdata = 'Q000'                 # Aグループ
        elif (c1 == 'B' or c1 == 'b') and len(kydata[0]) == 1:
            tdata = 'Q001'                 # Bグループ
        else:
            print('invalid data')
            continue

        # パルス幅を算出
        try:
            n = int(kydata[1], 10)         # チャンネル番号を整数に変換
        except:
            print('invalid data')
            continue
        if n > 11 or n < 0:
            print('invalid data')
            continue
        if len(kydata) == 2:
            if tdata == 'Q000':            # Aグループ
                pmch = n                   # パルス幅増減操作対象チャンネル設定のみ
            else:                          # Bグループ
                pbch = n                   # パルス幅表示チャンネル設定のみ
            continue
        try:
            s = int(kydata[2], 10)         # センターからの位置を整数に変換
        except:
            print('invalid data')
            continue

        if tdata == 'Q000':                # Aチャンネルのとき
            p = pwacent[n] + s             # 実際のパルス幅に変換
            if p < pwalow[n]:              # 最小値の制限
                p = pwalow[n]
            if p > pwahigh[n]:             # 最大値の制限
                p = pwahigh[n]
            pwalist[n] = p                 # パルス幅を変更
            pmch = n                       # パルス幅増減操作対象チャンネルを設定
            # 送信文字列準備
            tdata = tdata + format(n,'x') + ("{:03x}".format(pwalist[n])) + chr(0xd)

        else:                              # Bチャンネルのとき
            p = pwbcent[n] + s             # 実際のパルス幅に変換
            if p < pwblow[n]:              # 最小値の制限
                p = pwblow[n]
            if p > pwbhigh[n]:             # 最大値の制限
                p = pwbhigh[n]
            pwblist[n] = p                 # パルス幅を変更
            pbch = n                       # 表示チャンネルを設定
            # 送信文字列準備
            tdata = tdata + format(n,'x') + ("{:03x}".format(pwblist[n])) + chr(0xd)

        # パルス幅を送信
        readdata = FT_rpi.transfer_dacs(handle, tdata, 9)
        # 送信コマンドと応答を画面表示
        print('\r' + tdata[0:-1] + ' --> ' + readdata) 

    if c1 == '-':    # 最初の文字がマイナスキーはプログラム終了
        break

# デバイスをCLOSE
FT_rpi.close_dacs(handle)
